##
## new assembly postprocessing
## 
## handle global constructors
## handle cases where FP is not setup
##
## asmpp.awk (assembly postprocessing)
##

##
## the grammar of assembly files
##

##
## Common terminal symbols
## tk_eof                     : end of file
## Labels:
##  tk_label_asm_header        : speicial labels in header
##  tk_label_C                 : label def of a C symbol
##  tk_label_compiler          : label def of a compiler generated symbol
## Directives:
##  tk_directive_bss           : bss storage declaration (.common ...)
##  tk_directive_file          : .file
## 
## Instructions:
##  tk_insn_arith              : R = const
##  tk_insn_load               : SP-based load (R = SP[X])
##  tk_insn_store              : SP-based store (SP[X] = R)
##  tk_insn_special_proc       : special func
##  tk_insn_call               : call
##  tk_insn_return             : return
##  tk_insn_jmp                : jumps (except for calls)
##  tk_insn_inc_sp             : increment SP (SP += X)
##  tk_insn_setup_fp           : set FP in prologue (FP = SP + X)
##  tk_insn_reset_sp           : SP reset in epilogue opening (SP = FP + X)
##  tk_insn_free_frame         : free frame at the epilogue end (SP = FP)
##  tk_insn_free_frame+sp_load : free frame at the epilogue end (leave)
##                               also restore FP
##  tk_insn_other                 : anything else (non jumps)
## 
## Data
##  tk_datum                   : put a datum (.long, etc.)
## 
## Hints
##  tk_hint_prologue_begin     : hint that tells us the beginning of prologue
##  tk_hint_prologue_end       : hint that tells us the end of prologue
##  tk_hint_epilogue_begin     : hint that tells us the beginning of epilogue
##  tk_hint_epilogue_end       : hint that tells us the end of epilogue

## Nonteminals and syntax
## 
## asm_file ::=          asm_file_header (tk_label_C proc_or_data 
##                                      | tk_label_compiler 
##                                      | tk_datum 
##                                      | tk_string 
##                                      | tk_directive_bss)*
## 
## asm_file_header ::=   tk_directive_file tk_label_asm_header*
##
## proc_or_data ::=      (tk_label_compiler | tk_datum | tk_string)* 
##                        proc | tk_datum+ | tk_string
##                         prologue body_epilogue+
## proc ::=              prologue body_epilogue+
## body_epilogue ::=     body epilogue? | epilogue
##
## prologue ::=          prologue_head prologue_1* 
##                       (tk_hint_prologue_end | JMP prologue_1?)?
## prologue_1 ::=        (STORE|LOAD|OTHER|SETUP_FP|tk_label_compiler)
##
## prologue_head ::=     (machine/compiler dependent)
##
## body ::=              (STORE | LOAD | INC_SP | JMP | OTHER |
##                          | CALL | SPECIAL_PROC 
##                          | tk_label_compiler | tk_datum | tk_string)*
## epilogue ::=          (EPILOGUE_BEGIN_HINT OTHER*)? 
##                       RESET_SP?
##                       (LOAD | OTHER | tk_label_compiler)*
##                       epilogue_tail
##                       RETURN
##                       tk_label_compiler?
## epilogue_tail ::=     (compiler dependent)
##
## data_or_string ::=    tk_datum+ | tk_string
##

##
## ---------- basic grammar stuff ----------
##

##
## initialize FIRST set (change this if you change the above grammer)
##

function init_FIRST(\
		    changed, iter)
{
  if (C_dbg>0) dbg_print("init_FIRST");
# trailing space is important!!
  C_FIRST["asm_file"] = "";
  C_FIRST["asm_file_header"] = "tk_directive_file ";
  C_FIRST["proc"] = "";
  C_FIRST["proc_or_data"] = "tk_label_compiler tk_datum tk_string ";
  C_FIRST["body_epilogue"] = "";
  C_FIRST["prologue"] = "";
  C_FIRST["prologue_1"] = cat4("tk_insn_arith tk_insn_load tk_insn_store ",
			       "tk_insn_setup_fp tk_insn_other ",
			       "tk_label_compiler ",
			       "tk_directive_reorder tk_directive_noreorder ");
  C_FIRST["prologue_head"] = prologue_head_FIRST();
  C_FIRST["body"] = cat4("tk_insn_arith tk_insn_load tk_insn_store ",
			 "tk_insn_special_proc tk_insn_call tk_insn_jmp ",
			 "tk_insn_inc_sp tk_insn_other tk_label_compiler ",
			 "tk_directive_reorder tk_directive_noreorder tk_datum tk_string ");
  C_FIRST["epilogue"] = cat3("tk_hint_epilogue_begin ",
			     "tk_insn_load tk_insn_reset_sp ",
			     "tk_insn_free_frame tk_insn_free_frame+sp_load ");
  C_FIRST["free_frame"] = cat3("tk_insn_inc_sp ",
			       "tk_insn_free_frame ",
			       "tk_insn_free_frame+sp_load ");
  C_FIRST["data_or_string"] = "tk_datum tk_string ";

  if(C_dbg>0) {
    dbg_print("initial FIRST");
    for (x in C_FIRST) {
      dbg_print("FIRST(" x ") = " C_FIRST[x]);
    }
  }

  iter = 0;
  do {
    iter++;
    changed = 0;
    if (include_FIRST("prologue", "prologue_head")) changed = 1;
    if (include_FIRST("body_epilogue", "body")) changed = 1;
    if (include_FIRST("body_epilogue", "epilogue")) changed = 1;
    if (include_FIRST("proc", "prologue")) changed = 1;
    if (include_FIRST("proc_or_data", "proc")) changed = 1;
    if (include_FIRST("asm_file", "asm_file_header")) changed = 1;
  } while(changed);

  if(C_dbg>0){
    dbg_print("FIRST converged after " iter " iterations");
    for (x in C_FIRST) {
      dbg_print("FIRST(" x ") = " C_FIRST[x]);
    }
  }

  check_non_intersection_FIRST("proc", "data");
# we need rule out ambiguity that comes from ?
# currently we have one : sp_load found in body may start epilogue...
  if (C_dbg>0) dbg_print("init_FIRST done");
}

##
## make sure the grammar is an unambiguous LL1
## signal error if FIRST[x] and FIRST[y] have a common element
##

function check_non_intersection_FIRST(x, y)
{
  fx = C_FIRST[x];
  fy = C_FIRST[y];
  split(fy, a);
  for (i in a) {
    if (index(fx, a[i])) {
      fatal("grammar error: FIRST(" x ") and FIRST(" y ") have a common element " a[i]);
    }
  }
}

##
## force C_FIRST[x] >= C_FIRST[y] (in terms of set inclusion)
##

function include_FIRST(x, y,
		       fx, fy, a, c, i)
{
  fx = C_FIRST[x];
  fy = C_FIRST[y];
  split(fy, a);
  c = 0;
  for (i in a) {
    if (!index(fx, a[i])) {
      if(C_dbg>1)dbg_print("FIRST(" x ") += " a[i]);
# note that fx ends with space.
      fx = fx a[i] " ";
      c = 1;
    }
  }
  C_FIRST[x] = fx;
  if (c && C_dbg>1) {
    dbg_print("FIRST(" x ") == " fx);
  }
  return c;
}

##
## ---------- main, configuration, and initialization ----------
##

##
## main function
##

function main()
{
  common_init();
  if (C_task == "asmpp") {
    scan_asm_file();
  } else if (C_task == "tblpp") {
    scan_asm_file();
  } else if (C_task == "rem_prologue") {
    scan_asm_file_rem_prologue();
  } else wrong_task();
  task_finish();
}

##
## initialization
##

function common_init()
{
  if(C_dbg>0)dbg_print("common init");
# absolute constants
  C_errout = "/dev/stderr";
  C_stdout = "/dev/stdout";

# machine-specific parameters
  configure();
# parameters that should be given in the command line
  C_dbg = DBG;
  C_module = MODULE;
  C_task = TASK;

  init_machine_constants();

# constants
  C_tab_c = encode_file_name(FILENAME ".tab.c");
  C_proc_start_suffix = "_start_xxx";
  C_proc_epilogue_suffix = "_epilogue_xxx";
  C_proc_pure_epilogue_suffix = "_pure_epilogue_xxx";
  C_proc_end_suffix = "_end_xxx";
  C_proc_keep_frame_suffix = "_keep_frame_xxx";
  C_proc_info_suffix = "_st_proc_info_zzz__";
  C_fork_point_info_suffix = "_st_fork_point_info_zzz__";
  C_invalid_call_site_info_suffix = "_st_invalid_call_site_info_zzz__";
  C_data_info_suffix = "_st_data_info_zzz__";
  C_fork_block_start_prefix = "FPS";
  C_fork_block_end_prefix = "FPE";
  C_invalid_call_start_prefix = "ICS";
  C_invalid_call_end_prefix = "ICE";
# trailing space is important
  C_special_procedures =\
    cat5("__st_fork_block_start __st_fork_block_start0 ",
	 "__st_fork_block_end __st_fork_block_end0 ",
	 "__st_invalid_call_start __st_invalid_call_start0 ",
	 "__st_invalid_call_end __st_invalid_call_end0 ",
	 "st_dont_postprocess ");

  C_check_insn_count = 0;

# encoder
  init_proc_encoder();

# active stack frames
  G_stack_depth = 0;
  delete G_stack;
# number of procedures in this file
  G_n_procs = 0;
# number of named data in this file
  G_n_data = 0;
# number of fork points in this file
  G_n_fork_blocks = 0;
# number of invalid calls in this file
  G_n_invalid_calls = 0;
# procedures that never call StackThreads primitives
  delete G_unpostprocessed_procs;

# initialize FIRST set
  init_FIRST();

  L_tok = 0;
}

function init_proc_encoder()
{
  if (C_dbg>0) dbg_print("init_proc_encoder");
  delete C_conv;
  for (i = 0; i < 128; i++) {
    x = sprintf("%c", i);
    if (!match(x, /[_0-9A-Za-z]/) && match(x, /[[:print:]]+/) &&
	x != "'" && x != "=" && x != "<" && x != ">" && x != "`") {
      C_conv[i] = x;
    }
  }
  if (C_dbg>0) dbg_print("init_proc_encoder done");
}

function task_finish()
{
  if (C_task == "asmpp") {
    close(C_tab_c);
  } else if (C_task == "tblpp") {
  } else if (C_task == "rem_prologue") {
  } else wrong_task();
}

##
## set machine configuration parameters
##

function configure(\
		   a)
{
  if(C_dbg>0)dbg_print("configure");
  C_cpu = (ST_CPU_TYPE ? ST_CPU_TYPE : ENVIRON["ST_CPU_TYPE"]);
  C_os = (ST_OS_TYPE ? ST_OS_TYPE : ENVIRON["ST_OS_TYPE"]);
  C_compiler = (ST_COMPILER_TYPE ? ST_COMPILER_TYPE : ENVIRON["ST_COMPILER_TYPE"]);
  C_st_config = C_cpu " " C_os " " C_compiler;
  if (C_compiler == 0) {
    if (C_os == "winnt") {
      C_compiler = "cygwin";
    } else {
      C_compiler = "gcc";
    }
    warn("environment variable ST_COMPILER_TYPE is not set. set it to the default (" C_compiler ")");
  }
  C_supported_configs["i386","solaris","gcc"] = 1;
  C_supported_configs["i386","linux","gcc"] = 1;
  C_supported_configs["i386","winnt","cygwin"] = 1;
  C_supported_configs["sparc","solaris","gcc"] = 1;
  C_supported_configs["mips","irix","gcc"] = 1;
  C_supported_configs["alpha","osf1","gcc"] = 1;
  C_supported_configs["alpha","linux","gcc"] = 1;
  if (!(C_cpu SUBSEP C_os SUBSEP C_compiler in C_supported_configs)) {
    fatal("configuration " C_cpu " " C_os " " C_compiler " not supported");
  }
}

##
## initialize machine-specific parameters
##

function init_machine_constants()
{
  if (C_cpu == "i386") {
    C_md["comment_string"] = "/";
    C_md["RA"] = "RA";		# no register for return address
    C_md["FP"] = "%ebp";
    C_md["SP"] = "%esp";
    C_md["VSP"] = "%ecx";
    C_md["TLS"] = "%ebx";
    C_md["saved_regs"] = "%esi %edi %ebp";
    C_md["unrestored_saved_regs"] = "";
    C_md["call_delay_slot"] = 0;
    C_md["fix_gp_after_call"] = 0;
    C_md["ret_delay_slot"] = 0;
    C_md["RA_must_be_saved"] = 1;
    C_md["FP_must_be_saved"] = 1;
    C_md["FP_is_restored_in_tail"] = 1;
    C_md["min_sp_relative_size"] = 0;
    C_md["min_stack_align"] = 8;
    C_md["sign_to_grow_stack"] = "-";
    C_md["SP_must_be_reset"] = 0;
    C_md["restore_FP_after_free_frame"] = 1;
    if (C_os == "solaris" || C_os == "linux") {
      C_md["C_symbol_prefix"] = "";
      C_md["compiler_symbol_prefix"] = ".L";
      C_md_data_size[".byte"] = 1;
      C_md_data_size[".value"] = 2;
      C_md_data_size[".long"] = 4;
    } else if (C_os == "winnt") {
      C_md["C_symbol_prefix"] = "_";
      C_md["compiler_symbol_prefix"] = "L";
      C_md_data_size[".byte"] = 1;
      C_md_data_size[".word"] = 2;
      C_md_data_size[".long"] = 4;
    } else wrong_port("os");
  } else if (C_cpu == "sparc") {
    C_md["comment_string"] = "!";
    C_md["RA"] = "%o7";
    C_md["FP"] = "%i7";
    C_md["SP"] = "%sp";
    C_md["VSP"] = "%o5";
    C_md["TLS"] = "%l7";
    C_md["saved_regs"] = cat2("%o7 %i7 %i0 %i1 %i2 %i3 %i4 %i5 ",
			      "%l0 %l1 %l2 %l3 %l4 %l5 %l6 ");
    C_md["unrestored_saved_regs"] = "";
    C_md["call_delay_slot"] = 1;
    C_md["fix_gp_after_call"] = 0;
    C_md["ret_delay_slot"] = 1;
    C_md["RA_must_be_saved"] = 0;
    C_md["FP_must_be_saved"] = 0;
    C_md["FP_is_restored_in_tail"] = 0;
    C_md["min_sp_relative_size"] = 92;
    C_md["min_stack_align"] = 8;
    C_md["sign_to_grow_stack"] = "-";
    C_md["SP_must_be_reset"] = 1;
    C_md["restore_FP_after_free_frame"] = 0;
    if (C_os == "solaris") {
      C_md["C_symbol_prefix"] = "";
      C_md["compiler_symbol_prefix"] = ".L";
      C_md_data_size[".byte"] = 1;
      C_md_data_size[".uahalf"] = 2;
      C_md_data_size[".half"] = 2;
      C_md_data_size[".uaword"] = 2;
      C_md_data_size[".word"] = 2;
    } else if (C_os == "linux") {
      wrong_port("os");		# sparc linux
    } else wrong_port("os");
  } else if (C_cpu == "mips") {
    C_md["comment_string"] = "#";
    C_md["RA"] = "$31";
    C_md["FP"] = "$fp";
    C_md["SP"] = "$sp";
    C_md["TLS"] = "$23";
    C_md["VSP"] = "$25";
    C_md["saved_regs"] = cat2("$fp $31 $28 $16 $17 $18 $19 $20 $21 $22 ",
			      "$f20 $f22 $f24 $f26 $f28 $f30 ");
    C_md["unrestored_saved_regs"] = "$28 ";
    C_md["call_delay_slot"] = 0;
    C_md["fix_gp_after_call"] = 0;
    C_md["ret_delay_slot"] = 0;
    C_md["RA_must_be_saved"] = 0;
    C_md["FP_must_be_saved"] = 1;
    C_md["FP_is_restored_in_tail"] = 0;
    C_md["min_sp_relative_size"] = 16; # are you sure?
    C_md["min_stack_align"] = 8;
    C_md["sign_to_grow_stack"] = "-";
    C_md["SP_must_be_reset"] = 1;
    C_md["restore_FP_after_free_frame"] = 0;
    if (C_os == "irix") {
      C_md["C_symbol_prefix"] = "";
      C_md["compiler_symbol_prefix"] = "$L";
      C_md_data_size[".byte"] = 1;
      C_md_data_size[".half"] = 2;
      C_md_data_size[".word"] = 4;
    } else wrong_port("os");
  } else if (C_cpu == "alpha") {
    C_md["comment_string"] = "#";
    C_md["RA"] = "$26";
    C_md["FP"] = "$15";
    C_md["SP"] = "$30";
    C_md["TLS"] = "$14";
    C_md["VSP"] = "$25";
    C_md["saved_regs"] = cat2("$15 $26 $9 $10 $11 $12 $13 ", 
			      "$f2 $f3 $f4 $f5 $f6 $f7 $f8 $f9 ");
    C_md["unrestored_saved_regs"] = "";
    C_md["call_delay_slot"] = 0;
    C_md["fix_gp_after_call"] = 1;
    C_md["ret_delay_slot"] = 0;
    C_md["RA_must_be_saved"] = 1;
    C_md["FP_must_be_saved"] = 1;
    C_md["FP_is_restored_in_tail"] = 0;
    C_md["min_sp_relative_size"] = 0;
    C_md["min_stack_align"] = 8;
    C_md["sign_to_grow_stack"] = "-";
    C_md["SP_must_be_reset"] = 1;
    C_md["restore_FP_after_free_frame"] = 0;
    if (C_os == "osf1" || C_os == "linux") {
      C_md["C_symbol_prefix"] = "";
      C_md["compiler_symbol_prefix"] = "$";
      C_md_data_size[".byte"] = 1;
      C_md_data_size[".word"] = 2;
      C_md_data_size[".long"] = 4;
      C_md_data_size[".quad"] = 8;
    } else wrong_port("os");
  } else wrong_port("cpu");

  if (C_compiler == "gcc" || C_compiler == "cygwin") {
# the trailing space is important!!
    C_md["asm_header_labels"] =\
      "__gnu_compiled_c __gnu_compiled_cplusplus gcc2_compiled. ";
  } else wrong_port("compiler");

# check if all required symbols are defined
  C_md_comment_string = md_const("comment_string");
  C_md_C_symbol_prefix = md_const("C_symbol_prefix");
  C_md_compiler_symbol_prefix = md_const("compiler_symbol_prefix");
  C_md_RA = md_const("RA");
  C_md_FP = md_const("FP");
  C_md_SP = md_const("SP");
  C_md_VSP = md_const("VSP");
  C_md_TLS = md_const("TLS");
  C_md_saved_regs = md_const("saved_regs");
  C_md_unrestored_saved_regs = md_const("unrestored_saved_regs");
  C_md_ret_delay_slot = md_const("ret_delay_slot");
  C_md_call_delay_slot = md_const("call_delay_slot");
  C_md_fix_gp_after_call = md_const("fix_gp_after_call");
  C_md_RA_must_be_saved = md_const("RA_must_be_saved");
  C_md_FP_must_be_saved = md_const("FP_must_be_saved");
  C_md_FP_is_restored_in_tail = md_const("FP_is_restored_in_tail");
  C_md_min_sp_relative_size = md_const("min_sp_relative_size");
  C_md_min_stack_align = md_const("min_stack_align");
  C_md_sign_to_grow_stack = C_md["sign_to_grow_stack"];
  C_md_SP_must_be_reset = C_md["SP_must_be_reset"];
  C_md_restore_FP_after_free_frame = md_const("restore_FP_after_free_frame");
  C_md_asm_header_labels = md_const("asm_header_labels");
}

function md_const(x)
{
  if (x in C_md) {
    if (C_dbg>0) dbg_print("C_md_" x " = " C_md[x]);
    return C_md[x];
  }
  else {
    fatal("machine-dependent constant " x " undefined");
  }
}

##
## ---------- scanners (change them when you change grammar) ----------
##
## scanners are derived from grammar rules fairly mechanically.
## the token (line) type we are looking at is hold in variable L_tok.
##
## for rule A ::= B C D, we have:
##   scan_A() { 
##     if (L_tok is in FIRST(B)) scan_B(); else syntax_error();
##     if (L_tok is in FIRST(C)) scan_C(); else syntax_error();
##     if (L_tok is in FIRST(D)) scan_D(); else syntax_error();
##   }
## for rule A :: B | C | D, we have:
##   scan_A() { 
##     if (L_tok is in FIRST(B)) scan_B(); 
##     else if (L_tok is in FIRST(C)) scan_C(); 
##     else if (L_tok is in FIRST(D)) scan_D(); 
##     else syntax_error();
##   }
##

##
## asm_file ::=          asm_file_header (tk_label_C proc_or_data 
##                                      | tk_label_compiler 
##                                      | tk_datum 
##                                      | tk_string 
##                                      | tk_directive_bss)*

function scan_asm_file()
{
  enter_scanner("asm_file");
  peek_token();
  if (in_FIRST(L_tok, "asm_file_header")) {
    scan_asm_file_header();
  } else syntax_error(C_FIRST["asm_file_header"]);

# nop?
  if (L_tok == "tk_insn_other") {
    print_and_next_line(0);
  }

  while(1) {
    peek_token();
    if (L_tok == "tk_label_C") {
      action_label_C();
      scan_proc_or_data();
    } else if (L_tok == "tk_label_compiler") {
      action_label_compiler();
    } else if (L_tok == "tk_datum") {
      print_and_next_line(0);
    } else if (L_tok == "tk_string") {
      print_and_next_line(0);
    } else if (L_tok == "tk_directive_bss") {
      action_directive_bss();
    } else if (L_tok == "tk_directive_file") {
      action_directive_file();
    } else break;
  }
  peek_token();
  if (L_tok != "tk_eof") syntax_error("tk_eof");
  action_end_asm_file();
  leave_scanner("asm_file");
}

##
## asm_file ::= x* (tk_label_C x* tk_comment_real_insn_begins x*)*
##

function scan_asm_file_rem_prologue()
{
  enter_scanner("asm_file");
  
  while(1) {
    peek_token();
    if (L_tok != "tk_label_C" && L_tok != "tk_eof") {
      print_and_next_line(0);
    } else break;
  }
  while(1) {
    peek_token();
    if (L_tok == "tk_label_C") {
      print_and_next_line(0);
      while(1) {
	peek_token_commenting();
	if (L_tok == "tk_label_compiler" || L_tok == "tk_label_C") {
	  print_and_next_line(0);
	} else if (L_tok != "tk_comment_real_insn_begins" && 
		   L_tok != "tk_eof") {
	  comment_and_next_line("COMMENT OUT " $0);
	} else break;
      }
      peek_token();
      if (L_tok == "tk_comment_real_insn_begins") {
	print_and_next_line(0);
      } else {
	break;
        # Sep 13, 99
        # this was an error before. now simply exit the loop to handle a case
        # like this:
        # Lextext:
        # (EOF)
        # in this case, we are seeing EOF here.
        # syntax_error("tk_comment_real_insn_begins");
      }
      while(1) {
	peek_token();
	if (L_tok != "tk_comment_real_insn_ends" && 
	    L_tok != "tk_eof") {
	  print_and_next_line(0);
	} else break;
      }
      peek_token();
      if (L_tok == "tk_comment_real_insn_ends") {
	print_and_next_line(0);
      } else syntax_error("tk_comment_real_insn_ends");
      while(1) {
	peek_token();
	if (L_tok == "tk_label_compiler") {
	  print_and_next_line(0);
	} else if (L_tok != "tk_label_C" && L_tok != "tk_eof") {
	  comment_and_next_line("COMMENT OUT " $0);
	} else break;
      }
    } else break;
  }
  peek_token();
  if (L_tok != "tk_eof") syntax_error("tk_eof");
  leave_scanner("asm_file");
}

##
## asm_file_header ::=   tk_directive_file tk_label_asm_header
##

function scan_asm_file_header()
{
  enter_scanner("asm_file_header");
  peek_token();
  if (L_tok == "tk_directive_file") {
    action_directive_file();
  } else syntax_error("tk_directive_file");

  while(1) {
    peek_token();
    if (L_tok == "tk_label_asm_header") {
      action_label_asm_header();
    } else break;
  }

  leave_scanner("asm_file_header");
}

##
## proc_or_data ::=      (tk_label_compiler (tk_datum | tk_string))* 
##                        proc | tk_datum+ | tk_string
##

function scan_proc_or_data()
{
  enter_scanner("proc_or_data " G_last_C_label);
  while(1) {
    peek_token();
    if (L_tok == "tk_label_compiler") {
      print_and_next_line(0);
      while(1) {
	peek_token();
	if (L_tok == "tk_datum") {
	  print_and_next_line(0);
	} else if (L_tok == "tk_string") {
	  print_and_next_line(0);
	} else break;
      }
    } else break;
  }

  if (in_FIRST(L_tok, "proc")) {
    scan_proc();
  } else if (L_tok == "tk_datum") {
    action_aux_begin_data();
    while(1) {
      peek_token();
      if (L_tok == "tk_datum") {
	action_datum();
      } else break;
    }
  } else if (L_tok == "tk_string") {
    print_and_next_line(0);
  } 
  leave_scanner("proc_or_data " G_last_C_label);
}

function scan_proc(\
		   c)
{
  enter_scanner("proc " G_last_C_label);

  scan_prologue(); 
  c = 0;
  while(1) {
    peek_token();
    if (in_FIRST(L_tok, "body_epilogue")) {
      c++;
      scan_body_epilogue();
    } else {
      if (c == 0) syntax_error(C_FIRST["body_epilogue"]);
      break;
    }
  }
  action_aux_end_proc();
  leave_scanner("proc " G_last_C_label);
}

##
## prologue ::= prologue_head 
##              (STORE|LOAD|OTHER|SETUP_FP|tk_label_compiler)* 
##              tk_hint_prologue_end?
##

function scan_prologue()
{
  enter_scanner("prologue");
  action_aux_begin_prologue();
  scan_prologue_head();

  while(1) {
    peek_token();
    if (in_FIRST(L_tok, "prologue_1")) {
      scan_prologue_1();
    } else break;
  }

  peek_token();
  if (L_tok == "tk_insn_jmp") {
    print_and_next_line(0);
    if (G_branch_delays) {
      scan_prologue_1();
    }
  } else if (L_tok == "tk_hint_prologue_end") {
    print_and_next_line(0);
  }
    
  action_aux_end_prologue();
  leave_scanner("prologue");
}

function scan_prologue_head()
{
  enter_scanner("prologue_head");
  if (C_cpu == "i386") {
    i386_scan_prologue_head();
  } else if (C_cpu == "sparc") {
    sparc_scan_prologue_head();
  } else if (C_cpu == "mips") {
    mips_scan_prologue_head();
  } else if (C_cpu == "alpha") {
    alpha_scan_prologue_head();
  } else wrong_port("cpu");
  leave_scanner("prologue_head");
}

function prologue_head_FIRST()
{
  if (C_cpu == "i386") {
    return i386_prologue_head_FIRST();
  } else if (C_cpu == "sparc") {
    return sparc_prologue_head_FIRST();
  } else if (C_cpu == "mips") {
    return mips_prologue_head_FIRST();
  } else if (C_cpu == "alpha") {
    return alpha_prologue_head_FIRST();
  } else wrong_port("cpu");
}

##
## i386_prologue_head
## pushl %ebp; movl %esp,%ebp; (subl $XX,%ebp)?
##

function i386_prologue_head_FIRST()
{
  return "tk_insn_store ";
}

##
## i386 prologue_head ::= STORE (OTHER | tk_label_compiler)* FP_SETUP
##

function i386_scan_prologue_head()
{
# on i386, return address is already saved on the entry
  G_CP["saved_offset","RA"] = G_CP["SP"];
  G_CP["load_mnemonic","RA"] = "movl";
# pushl %ebp
  peek_token();
  if (L_tok == "tk_insn_store") {
    if (L_TI["src"] != C_md_FP || L_TI["base"] != C_md_SP) {
      syntax_error("pushl %ebp");
    }
    action_store_in_prologue();
  } else syntax_error("tk_insn_store");

  while(1) {
    peek_token();
    if (L_tok == "tk_insn_other") {
      print_and_next_line(0);
    } else if (L_tok == "tk_label_compiler") {
      print_and_next_line(0);
    } else break;
  }
# movl %esp,%ebp
  if (L_tok == "tk_insn_setup_fp") {
    action_setup_fp_in_prologue();
  } else syntax_error("tk_insn_setup_fp");

# subl $..,%esp
  peek_token();
  if (L_tok == "tk_label_compiler") {
    print_and_next_line(0);
  }
  peek_token();
  if (L_tok == "tk_insn_inc_sp") {
    if (L_TI["sign"] == "+") {
      new_sp = G_CP["SP"] + L_TI["offset"];
    } else if (L_TI["sign"] == "-") {
      new_sp = G_CP["SP"] - L_TI["offset"];
    } else wrong("wrong sign " L_TI["sign"]);
    if (C_dbg>0) {
      dbg_print("SP = " G_CP["SP"] " ==> " new_sp);
    }
    G_CP["SP"] = new_sp;
    print_and_next_line(0);
  } else if (match(L_tokens[0], /^movl \$[[:digit:]]+,%eax/)) {
# for large frames, a prologue looks like:
# movl $???,%eax
# call __alloca
    G_CP["large_stack_increment"] = substr(L_tokens[2,1], 2);
    print_and_next_line(0);
    peek_token();
    if (L_tok == "tk_insn_call" && L_TI["target"] == "__alloca") {
      if (C_dbg>0) {
	dbg_print("SP = " G_CP["SP"] " ==> " G_CP["SP"] + G_CP["large_stack_increment"]);
      }
      G_CP["SP"] -= G_CP["large_stack_increment"];
      print_and_next_line(0);
    }
  }
}

function sparc_prologue_head_FIRST()
{
  return "tk_hint_prologue_begin";
}

##
## sparc prologue head:
## (
## (1) add %sp,-??,%sp
## (2) set ???,%g1
##     sub %sp,%g1,%sp
## (3) sethi %hi(???),%g1
##     or %lo(???),%g1,%g1
##     sub %sp,%g1,%sp
##

function sparc_scan_prologue_head(\
				  s, r)
{
  peek_token();
  if (L_tok == "tk_hint_prologue_begin") {
    print_and_next_line(0);
  } else syntax_error("tk_hint_prologue_begin");

  r = 0;
  peek_token();
  if (L_tok == "tk_insn_arith" && L_TI["op"] == "+" && 
      is_number(L_TI["src",1]) && is_number(L_TI["src",2])) {
# set or sethi
    r = L_TI["dest"];
    v = L_TI["src",1] + L_TI["src",2];
    if(C_dbg>0) dbg_print(r " ==> " v);
    G_CP["const_reg",r] = v;
    print_and_next_line(0);
    peek_token();
# Sep 13, 99
# sethi %hi(??),?? is followed by or %lo(??),??,??
    if (match(L_flat_tokens[0], 
	      /^or[[:blank:]]%lo\([[:digit:]]+\),[[:blank:]]*%..,[[:blank:]]*%../)) {
      if (L_flat_tokens[3] != r || L_flat_tokens[4] != r) {
	syntax_error("or %lo(??)," r "," r);
      }
      split(L_flat_tokens[2], a, /\(|\)/);	# %lo(??) ==> %lo ??
      if (a[2] != v + a[2] % 4096) {
	syntax_error("or %lo(" (v + a[2] % 4096) ")," r "," r);
      }
      if(C_dbg>0) dbg_print(r " ==> " a[2]);
      G_CP["const_reg",r] = a[2];
      print_and_next_line(0);
    }
  }

  peek_token();
  if (L_tok == "tk_insn_inc_sp") {
# sub/add %sp,%..,%sp
    if (is_number(L_TI["offset"])) {
      if (L_TI["sign"] == "+") {
	o = 0 + L_TI["offset"];
      } else if (L_TI["sign"] == "-") {
	o = 0 - L_TI["offset"];
      } else wrong("wrong sign " L_TI["sign"]);
    } else if (r) {
      if (L_TI["offset"] != r) {
	wrong("wrong register " L_TI["offset"] " is used to grow stack (expected " r ")");
      }
      if (L_TI["sign"] == "+") {
	o = 0 + v;
      } else if (L_TI["sign"] == "-") {
	o = 0 - v;
      } else wrong("wrong sign " L_TI["sign"]);
    } else wrong("non-constant offset to grow stack (" L_TI["offset"] ")");
    if (C_dbg>0) dbg_print("SP ==> " G_CP["SP"] + o);
    G_CP["SP"] = G_CP["SP"] + o;
    print_and_next_line(0);
  } 
  # Sep 16 (gcc 2.95.1 may not generate prologue at all
  # else syntax_error("tk_insn_inc_sp");
}

##
## mips prologue header
## subu	$sp,$sp,40
## sw	$fp,28($sp)
## move	$fp,$sp
##

function mips_prologue_head_FIRST()
{
  return "tk_insn_inc_sp tk_directive_reorder tk_directive_noreorder ";
}

function mips_scan_prologue_head(\
				 v, r)
{
  while (1) {
    peek_token();
    if (L_tok == "tk_directive_reorder") {
      action_directive_reorder();
    } else if (L_tok == "tk_directive_noreorder") {
      action_directive_noreorder();
    } else break;
  }
  peek_token();
  if (L_tok == "tk_insn_inc_sp") {
    if (is_number(L_TI["offset"])) {
      if (L_TI["sign"] == "+") {
	o = 0 + L_TI["offset"];
      } else if (L_TI["sign"] == "-") {
	o = 0 - L_TI["offset"];
      } else wrong("wrong sign " L_TI["sign"]);
    } else wrong("non-constant offset to grow stack (" L_TI["offset"] ")");
    if (C_dbg>0) dbg_print("SP = " G_CP["SP"] " ==> " G_CP["SP"] + o);
    G_CP["SP"] = G_CP["SP"] + o;
    print_and_next_line(0);
  } else syntax_error("tk_insn_inc_sp");

# if frame size is large, it uses a temp register to address the location
# to save FP
# li $13,0x????????
# addu $13,$13,$sp
# sw $fp,??($13)
  peek_token();
  if (L_tok == "tk_label_compiler") {
    print_and_next_line(0);
  }
  peek_token();
  if (L_tok == "tk_insn_arith" && L_TI["op"] == "+" && 
      is_number(L_TI["src",1]) && is_number(L_TI["src",2])) {
    r = L_TI["dest"];
    v = L_TI["src",1] + L_TI["src",2];
    if(C_dbg>0) dbg_print(r " ==> " v);
    G_CP["const_reg",r] = v;
    print_and_next_line(0);
    peek_token();
    if (match(L_tokens[0], /^addu[[:space:]]+\$..?,\$..?,\$sp/)) {
      if (r == L_tokens[2,2]) {
	if(C_dbg>0) dbg_print(L_tokens[2,1] " ==> old_SP + " (G_CP["SP"] + v));
	G_CP["old_SP_relative_reg",L_tokens[2,1]] = G_CP["SP"] + v;
	print_and_next_line(0);
      } else syntax_error("addu ??," r ",$sp");
    }
  }
}

function alpha_prologue_head_FIRST()
{
# On OSF1, a procedure looks like this:
#   funcname:
#   { ldgp ... } /* optional */
#   funcname..ng:
#     inc SP,
# threfore, FIRST set is ldgp or tk_label_C.
# However, on Linux, the ..ng symbol begins with $.
#   funcname:
#   { ldgp ... } /* optional */
#   $funcname..ng:
#   inc SP,
# and therefore, if ldgp insn is missing, $funcname..ng is scanned by 
# proc_or_data. In this case, we must include tk_insn_inc_sp in the FIRST set.

  return "tk_insn_load_gp tk_label_C tk_insn_inc_sp ";
}

function alpha_scan_prologue_head(\
				  cr, n, sr, s, a)
{
  peek_token();
  if (L_tok == "tk_insn_load_gp") {
    print_and_next_line(0);
  }

  peek_token();
# a label like func_name..ng (OSF1) or $func_name..ng (Linux)
  if (L_tok == "tk_label_C" && L_TI["name"] == G_CP["name"] "..ng") {
    print_and_next_line(0);
  } else if (L_tok == "tk_label_compiler" && 
	     L_TI["name"] == make_compiler_symbol(G_CP["name"] "..ng")) {
    print_and_next_line(0);
  } 
# originally, it was considered as an error not to have ..ng: label here.
# however, on Alpha linux, this ..ng label is a compiler label (begins with $)
# and therefore may have been scanned by scan_proc_or_data.
# else syntax_error(G_CP["name"] "..ng:");

  peek_token();
  if (L_tok == "tk_insn_inc_sp") {
    if (C_dbg>0) dbg_print("alpha prologue pattern 1:");
    if(C_dbg>0) dbg_print("SP ==> " L_TI["offset"]);
    G_CP["SP"] = L_TI["offset"];
    print_and_next_line(0);
  } else if (match(L_tokens[0], /^stq \$31,-[[:digit:]]+\(\$30\)/)) {
    if (C_dbg>0) dbg_print("alpha prologue pattern 2:");
#	stq $31,-4096($30)
#	stq $31,-12288($30)
#	lda $30,-16400($30)
    while(1) {
      peek_token();
      if (match(L_tokens[0], /^stq \$31,-[[:digit:]]+\(\$30\)/)) {
	print_and_next_line(0);
      } else break;
    }
    peek_token();
    if (L_tok == "tk_insn_inc_sp") {
      if(C_dbg>0) dbg_print("SP ==> " L_TI["offset"]);
      G_CP["SP"] = L_TI["offset"];
      print_and_next_line(0);
    } else syntax_error("tk_insn_inc_sp");
  } else if (L_tok == "tk_insn_arith" && L_TI["op"] == "+" &&
	     is_number(L_TI["src",1]) && is_number(L_TI["src",2])) {
    if (C_dbg>0) dbg_print("alpha prologue pattern 3:");
#	addq $31,4,$5
#	lda $4,4096($30)
# bigframe_15..sc:
# 	stq $31,-8192($4)
#	subq $5,1,$5
#	lda $4,-8192($4)
#	bne $5,bigframe_15..sc
#	lda $30,-4112($4)

    cr = L_TI["dest"];
    n = L_TI["src",1] + L_TI["src",2];
    print_and_next_line(0);

# ldah R,XX(R)?
    peek_token();
    if (L_tok == "tk_insn_arith" && L_TI["dest"] == cr &&
	((L_TI["src",1] == cr && is_number(L_TI["src",2])) ||
	 (L_TI["src",2] == cr && is_number(L_TI["src",1])))) {
      if (L_TI["src",1] == cr) {
	n += L_TI["src",2];
      } else {
	n += L_TI["src",1];
      }
      print_and_next_line(0);
    }

    peek_token();
    if (match(L_tokens[0], /^lda \$..?,[[:digit:]]+\(\$30\)/)) {
      sr = L_TI["dest"];
      s = L_TI["src",1];
      print_and_next_line(0);
    } else syntax_error("lda $??,??($30)");
    peek_token();
    if (L_tok == "tk_label_C" && L_TI["name"] == G_CP["name"] "..sc") {
      print_and_next_line(0);
    } else if (L_tok == "tk_label_compiler" && 
	       L_TI["name"] == make_compiler_symbol(G_CP["name"] "..sc")) {
      print_and_next_line(0);
    } else syntax_error(G_CP["name"] "..sc:");
    peek_token();
    if (match(L_tokens[0], /^stq \$31,-[[:digit:]]+\(\$..?\)/) &&
	L_TI["base"] == sr) {
      print_and_next_line(0);
    } else syntax_error("stq $31,-??(" sr ")");
    peek_token();
    if (match(L_tokens[0], /^subq \$..?,1,\$..?/) && 
	L_TI["dest"] cr && L_TI["src",1] == cr) {
      print_and_next_line(0);
    } else syntax_error("subq " cr ",1," cr);
    peek_token();
    if (match(L_tokens[0], /^lda \$..?,-[[:digit:]]+\(\$..?\)/) &&
	L_TI["dest"] == sr && L_TI["src",2] == sr) {
      s += n * L_TI["src",1];
      print_and_next_line(0);
    } else syntax_error("lda " sr ",-??(" sr ")");
    peek_token();
    if (L_tok == "tk_insn_jmp" && 
	L_tokens[2,1] == cr && 
	(L_TI["target"] == G_CP["name"] "..sc" ||
	 L_TI["target"] == make_compiler_symbol(G_CP["name"] "..sc"))) {
      print_and_next_line(0);
    } else syntax_error("bne " cr "," G_CP["name"] "..sc");
    peek_token();
    if (match(L_tokens[0], /^lda \$30,-[[:digit:]]+\(\$..?\)/) &&
	L_TI["src",2] == sr) {
      if (C_dbg>0) dbg_print("SP ==> " s + L_TI["src",2]);
      G_CP["SP"] = s + L_TI["src",1];
      print_and_next_line(0);
    } else syntax_error("lda $30,-??(" sr ")");
  } else syntax_error("lda $30,-??($30) | stq $31,-??($30) | addq $31,??,??");
}

##
## prologue_1
##

function scan_prologue_1()
{
  enter_scanner("prologue_1");
  peek_token();
  if (L_tok == "tk_insn_arith") {
    print_and_next_line(0);
  } else if (L_tok == "tk_insn_load") {
    print_and_next_line(0);
  } else if (L_tok == "tk_insn_store") {
    action_store_in_prologue();
  } else if (L_tok == "tk_insn_setup_fp") {
    action_setup_fp_in_prologue();
  } else if (L_tok == "tk_insn_other") {
    print_and_next_line(0);
  } else if (L_tok == "tk_label_compiler") {
    print_and_next_line(0);
  } else if (L_tok == "tk_directive_reorder") {
    action_directive_reorder();
  } else if (L_tok == "tk_directive_noreorder") {
    action_directive_noreorder();
  } else syntax_error(cat2("tk_insn_arith tk_insn_load tk_insn_store ",
			   "tk_insn_setup_fp tk_insn_other tk_label_compiler"));
  leave_scanner("prologue_1");
}

##
## body_epilogue ::= body epilogue? | epilogue
##

function scan_body_epilogue()
{
  enter_scanner("body_epilogue");
  peek_token();
  if (in_FIRST(L_tok, "body")) {
    scan_body();
    peek_token();
    if (in_FIRST(L_tok, "epilogue")) {
      scan_epilogue();
    }
  } else if (in_FIRST(L_tok, "epilogue")) {
    scan_epilogue();
  }
  leave_scanner("body_epilogue");
}

## 
## body ::=              (OTHER | STORE | LOAD | INC_SP | JMP | CALL | SPECIAL_PROC 
##                          | tk_label_compiler | tk_datum | tk_string)*
##

function scan_body(\
		   ni)
{
  enter_scanner("body");

  reset_insn_count();

  while(1) {
    peek_token();
    if (L_tok == "tk_insn_arith") {
      tick_insn_count();
      action_arith_in_body();
    } else if (L_tok == "tk_insn_load") {
      tick_insn_count();
      print_and_next_line(0);
    } else if (L_tok == "tk_insn_store") {
      tick_insn_count();
      action_store_in_body();
    } else if (L_tok == "tk_insn_special_proc") {
      tick_insn_count();
      action_special_proc_1_in_body();
      if (C_md_fix_gp_after_call) {
	scan_fix_gp_after_special_proc();
      }
      if (C_md_call_delay_slot) {
	scan_delay_slot();
      }
      action_special_proc_2_in_body();
    } else if (L_tok == "tk_insn_call") {
      tick_insn_count();
      action_call_in_body();
      if (C_md_fix_gp_after_call) {
	scan_fix_gp();
      }
    } else if (L_tok == "tk_insn_jmp") {
      tick_insn_count();
      print_and_next_line(0);
    } else if (L_tok == "tk_insn_inc_sp") {
      tick_insn_count();
      action_inc_sp_in_body();
    } else if (L_tok == "tk_insn_other") {
      tick_insn_count();
      print_and_next_line(0);
    } else if (L_tok == "tk_label_compiler") {
      print_and_next_line(0);
    } else if (L_tok == "tk_directive_reorder") {
      action_directive_reorder();
    } else if (L_tok == "tk_directive_noreorder") {
      action_directive_noreorder();
    } else if (L_tok == "tk_datum") {
      print_and_next_line(0);
    } else if (L_tok == "tk_string") {
      print_and_next_line(0);
    } else break;
  }
  leave_scanner("body");
}

##
## delay slot after special procedure call
##

function scan_delay_slot()
{
  enter_scanner("delay_slot");
  while(1) {
    peek_token();
    if (L_tok == "tk_label_compiler") {
      print_and_next_line(0);
    } else break;
  }
  if (L_tok == "tk_insn_arith") {
    tick_insn_count();
    print_and_next_line(0);
  } else if (L_tok == "tk_insn_store") {
    tick_insn_count();
    action_store_in_body();
  } else if (L_tok == "tk_insn_load") {
    tick_insn_count();
    print_and_next_line(0);
  } else if (L_tok == "tk_insn_other") {
    tick_insn_count();
    print_and_next_line(0);
  } else syntax_error("tk_insn_store tk_insn_load tk_insn_arith tk_insn_other");
  leave_scanner("delay_slot");
}

##
## delay slot after special procedure call
##

function scan_fix_gp_after_special_proc()
{
  enter_scanner("fix_gp_after_special_proc");
  while(1) {
    peek_token();
    if (L_tok == "tk_label_compiler") {
      print_and_next_line(0);
    } else break;
  }
  if (L_tok == "tk_insn_load_gp") {
    comment_and_next_line(0);
  } else syntax_error("tk_insn_fix_gp");
  leave_scanner("fix_gp_after_special_proc");
}

##
## delay slot after special procedure call
##

function scan_fix_gp()
{
  enter_scanner("fix_gp");
  while(1) {
    peek_token();
    if (L_tok == "tk_label_compiler") {
      print_and_next_line(0);
    } else break;
  }
## we do not see load gp when the call is done by bsr
  if (L_tok == "tk_insn_load_gp") {
    tick_insn_count();
    print_and_next_line(0);
  }
  leave_scanner("fix_gp");
}

##
## epilogue ::=          (EPILOGUE_BEGIN_HINT (OTHER|tk_label_compiler)*)? 
##                       RESET_SP?
##                       (LOAD | OTHER | tk_label_compiler)*
##                       (FREE_FRAME|FREE_FRAME+SP_LOAD|INC_SP)  
##                       (LOAD | OTHER | tk_label_compiler)*
##                       RETURN
##                       tk_label_compiler?
## or 
## epilogue ::=          (EPILOGUE_BEGIN_HINT (OTHER|tk_label_compiler)*)? 
##                       RESET_SP?
##                       (LOAD | OTHER | tk_label_compiler)*
##                       RETURN
##                       (FREE_FRAME|FREE_FRAME+SP_LOAD|INC_SP)
##                       tk_label_compiler?
##

function scan_epilogue()
{
  enter_scanner("epilogue");
  action_aux_begin_epilogue();

  peek_token();
  if (L_tok == "tk_hint_epilogue_begin") {
    print_and_next_line(0);
    while (1) {
      peek_token();
      if (L_tok == "tk_insn_arith") {
	tick_insn_count();
	action_arith_in_epilogue();
      } else if (L_tok == "tk_label_compiler") {
	print_and_next_line(0);
      } else break;
    }
  }

  peek_token();
  if (L_tok == "tk_insn_reset_sp") {
    tick_insn_count();
    action_reset_sp_in_epilogue();
  }
  peek_token();
  if (L_tok == "tk_insn_offset_sp") {
    tick_insn_count();
    action_offset_sp_in_epilogue();
  }

  while(1) {
    peek_token();
    if (L_tok == "tk_insn_arith") {
      tick_insn_count();
      action_arith_in_epilogue();
    } else if (L_tok == "tk_insn_load") {
      tick_insn_count();
      action_load_in_epilogue();
    } else if (L_tok == "tk_insn_other") {
      tick_insn_count();
      print_and_next_line(0);
    } else if (L_tok == "tk_label_compiler") {
      print_and_next_line(0);
    } else {
      break;
    }
  }

  if (C_md_ret_delay_slot) {
# on machines where a return insn has a delay slot, a return insn must appear
# first and then we see an instruction that frees frame
    peek_token();
    if (L_tok == "tk_insn_return") {
      tick_insn_count();
      action_return_in_epilogue();
      peek_token();
      if (in_FIRST(L_tok, "free_frame")) {
	scan_free_frame();
      } else {
# Sep 16 (gcc 2.95.1 and probably newer may not move SP at all in 
# prologue/epilogue)
# syntax_error(C_FIRST["free_frame"]);
	print_and_next_line(0);
      }
    } else syntax_error("tk_insn_return");
  } else {
# on machines where a return insn does not have a delay slot, we see an
# instruction that frees a frame and then return. there may be intervening
# instructions between them
    peek_token();
    if (in_FIRST(L_tok, "free_frame")) {
      scan_free_frame();
    } else syntax_error(C_FIRST["free_frame"]);
    while(1) {
      peek_token();
      if (L_tok == "tk_insn_load") {
	if (C_md_restore_FP_after_free_frame && 
	    L_TI["base"] == C_md_SP && L_TI["dest"] == C_md_FP) {
# this applies to i386 epilogue in which a frame is freed and then fp is 
# restored, as in:
# movl %ebp,%esp
# popl %ebp
	  tick_insn_count();
	  print_and_next_line(0);
	} else sytnax_error("load after free frame is not FP = SP[X] type");
      } else if (L_tok == "tk_insn_other") {
	tick_insn_count();
	action_other_in_epilogue();
      } else if (L_tok == "tk_label_compiler") {
	print_and_next_line(0);
      } else break;
    }
    peek_token();
    if (L_tok == "tk_insn_return") {
      tick_insn_count();
      action_return_in_epilogue();
    } else syntax_error("tk_insn_return");
  }

  action_aux_almost_end_epilogue();

  peek_token();
  if (L_tok == "tk_label_compiler") {
    print_and_next_line(0);
  }
  leave_scanner("epilogue");
}

##
## scan an instruction that frees a frame
##

function scan_free_frame()
{
  enter_scanner("free_frame");
  if (L_tok == "tk_insn_inc_sp") {
# SP += frame_size on most CPUs
    tick_insn_count();
    action_inc_sp_in_epilogue();
  } else if (L_tok == "tk_insn_free_frame") { 
# i386 movl %ebp,%esp (SP = FP)
    tick_insn_count();
    action_free_frame_in_epilogue();
  } else if (L_tok == "tk_insn_free_frame+sp_load") {
# i386 leave insn
    tick_insn_count();
    action_free_frame_in_epilogue();
  } else syntax_error("tk_insn_inc_sp tk_insn_free_frame tk_insn_free_frame+sp_load");
  leave_scanner("free_frame");
}

##
## ---------- end of scanners ----------
##

##
## ---------- actions performed during scanning ----------
##

function action_directive_file()
{
  if (C_task == "asmpp") {
    print_and_next_line(0);
  } else if (C_task == "tblpp") {
    comment_and_next_line(0);
  } else wrong_task();
}

function action_label_asm_header()
{
  if (C_task == "asmpp") {
    print_and_next_line(0);
  } else if (C_task == "tblpp") {
    comment_and_next_line(0);
  } else wrong_task();
}

##
## action_label_C:
## record the name of the label so that we later know the function name
## or data name when we see the contents
##

function action_label_C(\
			cname)
{
  cname = C_symbol_name(L_TI["name"]);
  G_last_C_label = cname;
  if (C_task == "asmpp") {
# insert a label of start symbol
    print_(make_compiler_symbol(cname C_proc_start_suffix) ":");
  } else if (C_task == "tblpp") {
  } else wrong_task();
  print_and_next_line(0);
}

##
## action_label_compiler:
## called when text element begins with a compiler label
##

function action_label_compiler()
{
  if (C_task == "asmpp") {
    print_and_next_line(0);
  } else if (C_task == "tblpp") {
    name = unique_compiler_name(L_TI["name"]);
    print_and_next_line(name ":");
  } else wrong_task();
}

function action_directive_reorder()
{
  G_branch_delays = 0;
  print_and_next_line(0);
}

function action_directive_noreorder()
{
  G_branch_delays = 1;
  print_and_next_line(0);
}

##
## begin prologue.
## among other things, setup following four parameters.
## idx : a unique integer that identifies the procedure
## name : the name of the procedure
## SP : keeps track of the current SP - old SP (initially zero)
## FP : "?" keeps track of the current FP - old SP (initially "?")
## sp_relative_size : keeps track of maximum SP relative size (initially zero)
##

function action_aux_begin_prologue(\
				   n)
{
  if (C_task == "asmpp") {
    delete G_CP;
    n = G_n_procs;
    G_n_procs = n + 1;
    
    G_CP["idx"] = n;
    G_CP["name"] = G_last_C_label;
# these two parameters keep offset of SP and FP, relative to the SP at
# the entry to the procedure
    G_CP["SP"] = 0;
    G_CP["FP"] = "?";
    
    G_CP["sp_relative_size"] = C_md_min_sp_relative_size;
    G_CP["max_sp_shrink_size"] = 0;
    G_CP["fork_block_start_idx"] = G_n_fork_blocks;
    G_CP["n_fork_blocks"] = 0;
    G_CP["invalid_call_start_idx"] = G_n_invalid_calls;
    G_CP["n_invalid_calls"] = 0;
    
    G_CP["n_saved_regs"] = 0;
    
# do not postprocess global constructor call
    if (match(G_CP["name"], /_GLOBAL_/)) {
      if(C_dbg>0) dbg_print(G_CP["name"] " is a global constructor call (will not be postprocessed)");
      G_CP["postprocess"] = 0;
    } else {
      G_CP["postprocess"] = 1;
    }

# initially guess this is a leaf. made zero when we find a call in this proc
    G_CP["leaf"] = 1;
    G_CP["n_epilogues"] = 0;
    G_CP["n_pure_epilogues"] = 0;
    G_CP["n_other_after_free_frame"] = 0;
  } else if (C_task == "tblpp") {
  } else wrong_task();
}

##
## pre/post increment SP to keep track of the current offset of SP
## relative to the old value of SP 
##

function pre_increment_sp()
{
  if ("pre_inc_sp" in L_TI) {
    if (C_dbg) {
      dbg_print("SP = " G_CP["SP"] " ==> " G_CP["SP"] + L_TI["pre_inc_sp"]);
    }
    G_CP["SP"] += L_TI["pre_inc_sp"];
  }
}

function post_increment_sp()
{
  if ("post_inc_sp" in L_TI) {
    if (C_dbg) {
      dbg_print("SP = " G_CP["SP"] " ==> " G_CP["SP"] + L_TI["post_inc_sp"]);
    }
    G_CP["SP"] += L_TI["post_inc_sp"];
  }
}

##
## found a store instruction (which may be an i386 push 
## instruction) that may save a register in prologue. check if the register 
## is a saved register and it is saved for the first time in this prologue. 
## if so, record its offset from the old SP (the SP at the entry of the 
## current procedure) at G_CP["saved_offset",REG_NAME]. 
## otherwise, it is an argument store, so treated similary to an SP-relative
## store in the body
##

function action_store_in_prologue(src, o, ld,
				  sp, size)
{
  if (C_task == "asmpp") {
    old_sp = G_CP["SP"];
    pre_increment_sp();
    if (L_TI["base"] == C_md_SP || 
	("old_SP_relative_reg" SUBSEP L_TI["base"] in G_CP)) {
# base is SP or was derived from SP by r = SP + const
      src = L_TI["src"];
      if (L_TI["base"] == C_md_SP) {
	o = G_CP["SP"] + L_TI["offset"];
      } else {
	o = G_CP["old_SP_relative_reg",L_TI["base"]] + L_TI["offset"];
      }
      ld = L_TI["load_mnemonic"];
      size = L_TI["op_size"];
      if (size == 0) fatal("size == 0");
      if (index(C_md_saved_regs, src) && !("saved_offset" SUBSEP src in G_CP)) {
# this register is a saved register and stored via SP for the first 
# time in this prologue.
# record the fact that this register is saved and its offset
	if (C_dbg>0) {
	  dbg_print(src " saved at oldSP[" o "] = SP[" (o - G_CP["SP"]) "]");
	}
	G_CP["saved_reg",G_CP["n_saved_regs"]] = src;
	G_CP["saved_offset",src] = o;
	G_CP["load_mnemonic",src] = ld;
	G_CP["n_saved_regs"]++;
      } else {
# this instruction will be an argument store
	if (C_dbg>0) {
	  dbg_print(src " stored at SP[" (o - G_CP["SP"]) "] as an argument");
	}
	if (!("base_SP" in G_CP)) {
# if we have seen an argument store, assume SP has already been fixed
	  if(C_dbg>0)dbg_print("base_SP <= " G_CP["SP"] " (fixed)");
	  G_CP["base_SP"] = old_sp;
	}
	record_arg_store((o - G_CP["SP"]) + size);
      }
    }
    post_increment_sp();
  } else if (C_task == "tblpp") {
  } else wrong_task();
  print_and_next_line(0);
}

function action_setup_fp_in_prologue()
{
  if ("fp_setup" in G_CP) {
    wrong("FP is setup twice in " G_CP["name"]);
  }

  if (is_number(L_TI["offset"])) {
    if (L_TI["sign"] == "+") {
      o = 0 + L_TI["offset"];
    } else if (L_TI["sign"] == "-") {
      o = 0 - L_TI["offset"];
    } else wrong("wrong sign " L_TI["sign"]);
  } else if ("const_reg" SUBSEP L_TI["offset"] in G_CP) {
    if (L_TI["sign"] == "+") {
      o = 0 + G_CP["const_reg",L_TI["offset"]];
    } else if (L_TI["sign"] == "-") {
      o = 0 - G_CP["const_reg",L_TI["offset"]];
    } else wrong("wrong sign " L_TI["sign"]);
  } else wrong("expected a constant offset at setup fp");
  if (C_dbg>0) {
    dbg_print("FP ==> " G_CP["SP"] + o);
  }
  G_CP["FP"] = G_CP["SP"] + o;
  G_CP["fp_setup"] = 1;
  print_and_next_line(0);
}

##
## end prologue. set following parameters in G_CP.
## return_address_fp_offset : the offset where return address is saved, 
## relative to FP.
## parent_fp_fp_offset : the offset where the parent FP is saved,
## relative to FP.
## base_SP-FP : SP - FP at this point
##

function action_aux_end_prologue()
{
  if (C_task == "asmpp") { 
# record where things were saved in terms of OFFSET FROM FP.
# we have recorded offsets from the SP. we translate them into 
# FP-relative offsets as follows.
# X - FP = (X - oldSP) + (oldSP - FP) = (X - oldSP) - (FP - oldSP)
    if ("saved_offset" SUBSEP C_md_RA in G_CP) {
      G_CP["return_address_fp_offset"] = \
	G_CP["saved_offset",C_md_RA] - G_CP["FP"];
    } else {
      if (C_md_RA_must_be_saved) {
	fatal("saved_offset," C_md_RA " undefined in " G_CP["name"] " prologue");
      }
    }
    if ("saved_offset" SUBSEP C_md_FP in G_CP) {
      G_CP["parent_fp_fp_offset"] = \
	G_CP["saved_offset",C_md_FP] - G_CP["FP"];
    } else {
      if (C_md_FP_must_be_saved) {
	fatal("saved_offset," C_md_FP " undefined in " G_CP["name"] " prologue");
      }
    }
# FP must be setup if and only if it is saved
    if (("saved_offset" SUBSEP C_md_FP in G_CP) && !("fp_setup" in G_CP)) {
      wrong("FP is saved, yet it is not set up");
    } else if (!("saved_offset" SUBSEP C_md_FP in G_CP) && ("fp_setup" in G_CP)) {
      wrong("FP is not saved, yet it is set up");
    }
    
# save SP at the end of prologue
# SP - FP (it may have been set in prologue)
    if (!("base_SP" in G_CP)) {
      G_CP["base_SP"] = G_CP["SP"];
    }
    G_CP["base_SP-FP"] = G_CP["base_SP"] - G_CP["FP"];
    
#
    if (C_dbg>0) {
      dbg_print("---- summary of " G_CP["name"] " prologue");
      if ("return_address_fp_offset" in G_CP) {
	dbg_print(cat3("  return_address_fp_offset = ",
		       G_CP["return_address_fp_offset"] " = ",
		       G_CP["saved_offset",C_md_RA] " - " G_CP["FP"]));
      } else {
	dbg_print("  return_address_fp_offset = undefined");
      }
      if ("parent_fp_fp_offset" in G_CP) {
	dbg_print(cat3("  parent_fp_fp_offset = ",
		       G_CP["parent_fp_fp_offset"] " = ",
		       G_CP["saved_offset",C_md_FP] " - " G_CP["FP"]));
      } else {
	dbg_print("  parent_fp_fp_offset = undefined");
      }
      dbg_print(cat3("  base_SP-FP = ", 
		     G_CP["base_SP-FP"] " = ", 
		     G_CP["SP"] " - " G_CP["FP"]));
      dbg_print("---- END summary of " G_CP["name"] " prologue");
    }
  } else if (C_task == "tblpp") {
  } else wrong_task();
}

##
## record the maximum offset
##

function record_arg_store(sp_offset)
{
  if (sp_offset > G_CP["sp_relative_size"]) {
    if (C_dbg>0) {
      dbg_print("sp_relative_size " G_CP["sp_relative_size"] " ==> " sp_offset);
    }
    G_CP["sp_relative_size"] = sp_offset;
  }
}

##
## actions in body
##

## arith

function is_effectively_const_operand(x)
{
  return (is_number(x) || "const_reg" SUBSEP x in G_CP);
}

function const_operand_val(x)
{
  if (is_number(x)) return x;
  else if ("const_reg" SUBSEP x in G_CP) return G_CP["const_reg",x];
  else fatal(x " is not an effectively const operand");
}

function emulate_arith(\
		       x, y)
{
  x = L_TI["src",1];
  y = L_TI["src",2];
  if (is_effectively_const_operand(x) && is_effectively_const_operand(y)) {
    vx = const_operand_val(x);
    vy = const_operand_val(y);
    if (L_TI["op"] == "+") {
      if(C_dbg>0) dbg_print(L_TI["dest"] " ==> " vx " + " vy " = " (vx + vy));
      G_CP["const_reg",L_TI["dest"]] = vx + vy;
    } else if (L_TI["op"] == "-") {
      if(C_dbg>0) dbg_print(L_TI["dest"] " ==> " vx " - " vy " = " (vx - vy));
      G_CP["const_reg",L_TI["dest"]] = vx - vy;
    } else wrong("wrong op " L_TI["op"]);
  } else {
    delete G_CP["const_reg",L_TI["dest"]];
  }
}

function action_arith_in_body()
{
  emulate_arith();
  print_and_next_line(0);
}

##
## found an SP-relative store instruction in procedure body.
##

function action_store_in_body(\
			      size)
{
  if (C_task == "asmpp") {
    pre_increment_sp();
    if (L_TI["base"] == C_md_SP) {
# we must actually deal with non-SP-based stores where the base register was
# derived from SP. we ignore them for now
      src = L_TI["src"];
      if (L_TI["base"] == C_md_SP) {
	o = G_CP["SP"] + L_TI["offset"];
      } else {
	fatal("should not happen");
      }
      if (C_dbg>0) {
	dbg_print(src " stored at SP[" (o - G_CP["SP"]) "] as an argument");
      }
      size = L_TI["op_size"];
      if (size == 0) fatal("size == 0");
      record_arg_store((o - G_CP["SP"]) + size);
    }
    post_increment_sp();
  } else if (C_task == "tblpp") {
  } else wrong_task();
  print_and_next_line(0);
}

##
## call
##

function action_call_in_body(\
			     t, n, s)
{
  t = L_TI["target"];
  if (symbol_type(t) == "C") {
    n = C_symbol_name(t);
  } else if (symbol_type(t) == "compiler") {
    n = compiler_symbol_name(t);
  } else {
    n = t;
  }
  n = symbol_stem(n);
  if (!(n in G_unpostprocessed_procs) && n != G_CP["name"]) {
    G_CP["leaf"] = 0;
  }
  print_and_next_line(0);
}

# if t is a symbol derived from a C symbol name (like xxx..ng on the Alpha),
# return its stem (xxx). return 0 otherwise.
function symbol_stem(n)
{
# handle "xxx..ng" ==> xxx
  if (C_cpu == "alpha") {
    if (match(n, /\.\.ng$/)) {
      l = length(n);
      return substr(n, 1, l - length("..ng"));
    } else {
      return n;
    }
  } else if (C_cpu == "i386") {
    return n;
  } else if (C_cpu == "sparc") {
    return n;
  } else if (C_cpu == "mips") {
    return n;
  } else wrong_port("cpu");
}


##
## found an instruction that increments SP
##

function action_inc_sp_in_body()
{
  if (C_task == "asmpp") {
    o = L_TI["offset"];
    if (o > G_CP["max_sp_shrink_size"]) {
      if (C_dbg>0) {
	dbg_print("max_sp_shrink_size " G_CP["max_sp_shrink_size"] " ==> " o);
      }
      G_CP["max_sp_shrink_size"] = stack_align(o, 1);
    } else {
      if (C_dbg>0) {
	dbg_print("max_sp_shrink_size stays at " G_CP["max_sp_shrink_size"]);
      }
    }
  } else if (C_task == "tblpp") {
  } else wrong_task();
  print_and_next_line(0);
}

##
## if fraction = 1, 
##   return the minimum multiple of C_md_min_stack_align that is greater 
##   than or equal to X.
## if fraction = -1,
##   return the maximum multiple of C_md_min_stack_align that is smaller
##   than or equal to X.
##

function stack_align(x, fraction)
{
  m = x % C_md_min_stack_align;
  if (m == 0) return x;
  else if (fraction == 1) {
    return x + (C_md_min_stack_align - m);
  } else return x - m;
}

##
## special procedure
##

function action_special_proc_1_in_body(\
				       t, n, l, idx)
{
  if (C_task == "asmpp") {
    t = L_TI["target"];
    if (symbol_type(t) != "C") {
      fatal("special proc that is not C symbol (" t ")");
    }
    n = C_symbol_name(t);
    if (n == "st_dont_postprocess") {
      G_CP["postprocess"] = 0;
    } else if (match(n, 
		     /^__st_(fork_block|invalid_call)_(start|end)0?$/)) {
      if (match(n, /fork_block/)) {
	idx = G_n_fork_blocks;
	if (match(n, /start/)) {
	  l = C_fork_block_start_prefix;
	} else if (match(n, /end/)) {
	  l = C_fork_block_end_prefix;
	  G_CP["n_fork_blocks"]++;
	  G_n_fork_blocks = idx + 1;
	} else fatal(n "does not match either with start|end");
      } else if (match(n, /invalid_call/)) {
	idx = G_n_invalid_calls;
	if (match(n, /start/)) {
	  l = C_invalid_call_start_prefix;
	} else if (match(n, /end/)) {
	  l = C_invalid_call_end_prefix;
	  G_n_invalid_calls = idx + 1;
	  G_CP["n_invalid_calls"]++;
	} else fatal(n "does not match either with start|end");
      } else {
	fatal(n " does not match either with fork_block|invalid_call");
      }
      G_CP["label_to_insert"] = C_md_compiler_symbol_prefix l idx;
    } else wrong("unknown special procedure " n);
    comment_and_next_line(0);
  } else if (C_task == "tblpp") {
    print_and_next_line(0);
  } else wrong_task();
}

##
## insert label after delay slot of special procedure calls
##

function action_special_proc_2_in_body()
{
  if ("label_to_insert" in G_CP) {
    print_(G_CP["label_to_insert"] ":");
    delete G_CP["label_to_insert"];
  }
}

##
## begin epilogue
##

function action_aux_begin_epilogue(\
				   n)
{
  if (C_task == "asmpp") {
    n = G_CP["n_epilogues"];
    print_(C_md_compiler_symbol_prefix G_CP["name"] C_proc_epilogue_suffix n ":");
  } else if (C_task == "tblpp") {
  } else wrong_task();
}

##
## basic block instruction appearing in epilogue
##

function action_arith_in_epilogue()
{
  emulate_arith();
  print_and_next_line(0);
}

##
## reset SP at the beginning of epilogue
##

function action_reset_sp_in_epilogue(\
				     o, l)
{
  if (C_task == "asmpp") {
# comment out the original instruction that resets SP.
    if (G_CP["postprocess"] && !G_CP["leaf"]) {
      if (is_number(L_TI["offset"])) {
	if (L_TI["sign"] == "+") {
	  o = 0 + L_TI["offset"];
	} else if (L_TI["sign"] == "-") {
	  o = 0 - L_TI["offset"];
	} else fatal("wrong sign " L_TI["sign"] " for insn");
      } else if ("const_reg" SUBSEP L_TI["offset"] in G_CP) {
	if (L_TI["sign"] == "+") {
	  o = 0 + G_CP["const_reg",L_TI["offset"]];
	} else if (L_TI["sign"] == "-") {
	  o = 0 - G_CP["const_reg",L_TI["offset"]];
	} else fatal("wrong sign " L_TI["sign"] " for insn");

	G_CP["restore_base_offset_reg"] = L_TI["offset"];
	G_CP["restore_base_offset_reg_sign"] = L_TI["sign"];

      } else wrong("expected a constant offset at reset sp");
      G_CP["saved_reset_sp"] = $0;
# simulate the effect of this instruction
      if (C_dbg>0) {
	dbg_print("SP = " G_CP["SP"] " ==> " G_CP["FP"] + o);
      }
      G_CP["SP"] = G_CP["FP"] + o;
# SP must be set to the same point as has been set at the end of prologue
      if (G_CP["SP"] != G_CP["base_SP"]) {
	wrong(cat3("SP restored to ",
		   "old_SP + " G_CP["SP"] " at epilogue, while set to ",
		   "old_SP + " G_CP["base_SP"] " at prologue"));
      }
      comment_and_next_line(0);
    } else print_and_next_line(0);
  } else if (C_task == "tblpp") {
    print_and_next_line(0);
  } else wrong_task();
}

##
## action_offset_sp_in_epilogue: 
## recognizes an instruction that does "R = SP + X" or "R = SP - X" in 
## epilogue.
## this currently deals with mips epilogues for large frames.
## mips epilogues for small frames use SP as the base, as in:
##   move $sp,$fp
##   lw R,O($sp)
##   lw R,O($sp)
##   ...
## for large frames, mips epilogue uses a separate register that addresses
## locations where registers are saved, because they are too far from SP.
## so, an epilogue looks like:
##   move $sp,$fp
##   addu $13,$12,$sp <---
##   lw R,O($13)
##   lw R,O($13)
## we are now looking at the instruction pointed to by "<---" above.
## seeing an instruction that matches "B = SP + S" or "B = SP - S", we assume 
## that R is going to be used as the base register for restoring saved 
## registers. we record this here. we also record the offset (from the old SP) 
## of the established base (G_CP["restore_base_offset"]).
## when S is not a literal constant, we assume that we have seen an instruction
## that loads S with a constant. such instruction must be duplicated in the
## pure epilogue, so we record S and sign ("+" or "-", depending on whether
## R = SP + S or R = SP - S.

function action_offset_sp_in_epilogue()
{
  G_CP["restore_base"] = L_TI["dest"];
  G_CP["reset_restore_base"] = 1;
  if (is_number(L_TI["offset"])) {
    G_CP["restore_base_offset"] = G_CP["SP"] + L_TI["offset"];
  } else if ("const_reg" SUBSEP L_TI["offset"] in G_CP) {
    G_CP["restore_base_offset"] =\
      G_CP["SP"] + G_CP["const_reg",L_TI["offset"]];
    G_CP["restore_base_offset_reg"] = L_TI["offset"];
    G_CP["restore_base_offset_reg_sign"] = L_TI["sign"];
  } else wrong("non const offset");
  if(C_dbg>0) dbg_print("restore base: " L_TI["dest"] " ==> " G_CP["old_SP_relative_reg",L_TI["dest"]]);
# since this instruction uses SP, and reset SP has been commented out,
# we must establish a virtual SP.
  s = make_offset_pointer(C_md_FP, "+", 
			  G_CP["base_SP"] - G_CP["FP"], C_md_VSP);
  print_("	" s);
	 
# and derive restore base from vsp
  s = make_offset_pointer(C_md_VSP, L_TI["sign"], 
			  L_TI["offset"], G_CP["restore_base"]);
  print_and_next_line("	" s);
}

##
## action_load_in_epilogue:
##
## if the procedure is designated not to postprocess or the procedure is a
## leaf, simply print it.
## for postprocessed non-leaf procedures,
## check if it looks like an instruction that restores a register saved in
## prologue. if it does, replace it with an instruction that uses the correct
## base register. the correct base register is a virtual SP if the original
## instruction uses SP. otherwise it is just the base register of the original
## instruction. 
##

function action_load_in_epilogue(\
				 sp, o, d)
{
  if (C_task == "asmpp") {
# modify SP if this is a pop
    pre_increment_sp();
    if (G_CP["postprocess"] && !G_CP["leaf"]) {
# this is not leaf and postprocess is not turned off.
# make sure the prologue has set up FP
      if ("fp_setup" in G_CP) {
# check if dest register has been saved in prologue
	d = L_TI["dest"];
	if ("saved_offset" SUBSEP d in G_CP) {
	  if ("restored" SUBSEP d in G_CP) wrong(d " is restored twice");
	  fix_restore_base();
	  check_restore_base(L_TI["base"]);
	  comment_and_next_line(0);
	  if (d == G_CP["restore_base"]) {
	    wrong("do not destroy restore base!!");
	  }
	  if (d != C_md_FP) {
# we must not destroy FP here because we use it to check if the frame 
# can be freed
	    if (L_TI["base"] == C_md_SP) {
# treat SP-based stores specially because SP may be altered by push/pops
	      o = G_CP["SP"] + L_TI["offset"];
	    } else {
	      o = G_CP["restore_base_offset"] + L_TI["offset"];
	    }
	    G_CP["restored",d] = 1;
	    print_restore_reg(L_TI["load_mnemonic"], o, d, 0);
	  }
	} else {
# the destination is not a saved register
	  print_and_next_line(0);
	}
      } else {
# this procedure is not leaf, yet FP is not setup for this procedure.
# StackThreads/MP does not permit such procedures.
# if you are really sure that this procedure does not block, you can just
# print it out. but we are currently on the side of safety
	wrong("procedure " G_CP["name"] " is not a leaf, yet does not have FP");
      }
    } else {
# this procedure is leaf, so we do not postprocess it
      print_and_next_line(0);
    }
    post_increment_sp();
  } else if (C_task == "tblpp") {
    print_and_next_line(0);
  } else wrong_task();
}

function fix_restore_base()
{
  if (!("restore_base" in G_CP)) {
# if we have not seen offset_sp at the epilogue beginning, the base register
# is VSP
    if(C_dbg>0) dbg_print("default restore base: " C_md_VSP " ==> " G_CP["base_SP"]);
    G_CP["restore_base"] = C_md_VSP;
    G_CP["restore_base_offset"] = G_CP["base_SP"];
  }
}

##
## check_restore_base(b):
## called when we see an instruction that restores a saved register in an
## epilogue sequence. B is the base register of the load instruction.
## check if B matches, if any, expected base register. we expect a base
## register to be R either (1) when we have seen an instruction that derives R 
## from SP (R = SP +(-) X) where X is a literal constant or a register that 
## contains a constant (see action_offset_sp), or (2) when we have already 
## seen a load instruction that restores a saved register in this epilogue 
## (i.e., an epilogue must use a single base register throughly).
##

function check_restore_base(b)
{	
  if (!("restore_base" in G_CP)) wrong("restore_base undefined");
  if (b != G_CP["restore_base"]) {
    if (G_CP["restore_base"] == C_md_VSP && b == C_md_SP) {
# OK
    } else {
      wrong(cat2("unexpected base register " b " (expected ",
		 G_CP["restore_base"] ")"));
    }
  }
}

##
## print an instruction that restores R from old SP + O, using LD to load R.
## PURE is 1 if this is used in a pure epilogue.
## use G_CP["restore_base"] as the base register.
## 

function print_restore_reg(ld, o, r, pure,
			   b, offs, offs_reg, sign, s)
{
# check if this register has been saved
  if (!("saved_offset" SUBSEP r in G_CP)) {
    wrong(r " restored in epilogue but not saved in prologue of " G_CP["name"]);
  }
# check if the offset is consistent with where the register has been saved
  if (G_CP["saved_offset",r] != o) {
    wrong(r " is restored from wrong offset (" o ", expected " G_CP["saved_offset",r] ")");
  }
# check if the ld is consistent with where the register has been saved
  if (G_CP["load_mnemonic",r] != ld) {
    wrong(cat2(r " is restored using unexpected load insn (" ld ", expected ",
	       G_CP["load_mnemonic",r] ")"));
  }
  if (G_CP["leaf"]) {
    fatal("leaf should not use this!!");
  } else if ("saved_reset_sp" in G_CP || !C_md_SP_must_be_reset) {
# in this case FP has been setup for this procedure, so FP is valid.
    if (!("restore_base" in G_CP)) {
      wrong("print_restore_reg : restore_base is not defined");
    }
    b = G_CP["restore_base"];
# establish restore base if it has not been established. print an instruction
# R = FP + O, where R is the restore base and O is the appropriate offset
# from FP.
# this is done once or twice per (postprocessed) procedure. first, if the 
# real epilogue uses SP as the restore base, the first SP-based load
# explicitly establishes virtual SP. second, the flag "reset_restore_base" 
# is explicitly turned off just before a pure epilogue, so the first 
# SP-based load in a pure epilogue reestablishes it.
    if (!("reset_restore_base" in G_CP)) {
      offs = G_CP["restore_base_offset"] - G_CP["FP"];
      if ("restore_base_offset_reg" in G_CP) {
	offs_reg = G_CP["restore_base_offset_reg"];
	sign = G_CP["restore_base_offset_reg_sign"];
	if (sign == "-") {
	  if (pure) {
	    print_("	" make_const(0 - offs, offs_reg));
	  }
	  print_("	" make_offset_pointer(C_md_FP, "-", offs_reg, b));
	} else {
	  if (pure) {
	    print_("	" make_const(offs, offs_reg));
	  }
	  print_("	" make_offset_pointer(C_md_FP, "+", offs_reg, b));
	}
      } else {
	print_("	" make_offset_pointer(C_md_FP, "+", offs, b));
      }
# set it so that we do not reset it twice
      G_CP["reset_restore_base"] = 1;
    }
    s = make_load(ld, b, o - G_CP["restore_base_offset"], r);
  } else {
    wrong("saved_reset_sp is not defined for non-leaf procedure " G_CP["name"]);
  }
  print_("	" s);
}


##
## check_free_frame
##
## if (SP > FP) goto keep_frame;
## if (tls[0] <= FP) goto keep_frame;
## > and <= become reverse when stack grows upwards.
## > and <= must be done in unsinged
## 

function check_free_frame(Lkf)
{
  if (C_cpu == "i386") {
    i386_check_free_frame(Lkf);
  } else if (C_cpu == "sparc") {
    sparc_check_free_frame(Lkf);
  } else if (C_cpu == "mips") {
    mips_check_free_frame(Lkf);
  } else if (C_cpu == "alpha") {
    alpha_check_free_frame(Lkf);
  } else wrong_port("cpu");
}

function i386_check_free_frame(Lkf)
{
  print_("	/ *tls - %ebp");
  print_("	cmpl %ebp,0(%ebx)");
  print_("	/ if (*tls <= ebp) goto " Lkf);
  print_("	jbe " Lkf);
  print_("	/ %esp - %ebp");
  print_("	cmpl %ebp,%esp");
  print_("	/ if (%esp > %ebp) goto " Lkf);
  print_("	ja " Lkf);
}

function sparc_check_free_frame(Lkf)
{
  print_("	! %o3 = *tls");
  print_("	ld [%l7],%o3")
  print_("	! %i7 - %sp");
  print_("	cmp %i7,%sp");
  print_("	! if (%i7 < %sp) goto " Lkf);
  print_("	blu " Lkf);
  print_("	! %i7 - *tls");
  print_("	cmp %i7,%o3");
  print_("	! if(%i7 >= *tls) goto " Lkf);
  print_("	bgeu " Lkf);
  print_("	nop");
}

function mips_check_free_frame(Lkf)
{
  print_("	# $5 = *tls");
  print_("	lw	$5,0($23)");
  print_("	# $6 = ($fp < *tls)");
  print_("	sltu	$6,$fp,$5");
  print_("	# if ($fp >= *tls) goto " Lkf);
  print_("	beq	$6,$0," Lkf);
  print_("	nop");
  print_("	# $5 = ($fp < $sp)"); 
  print_("	sltu	$5,$fp,$sp");
  print_("	# if ($fp < $sp) goto " Lkf);
  print_("	bne	$5,$0," Lkf);
  print_("	nop");
}

function alpha_check_free_frame(Lkf)
{
  print_("	# $1 = *tls");
  print_("	ldq $1,0($14)");
  print_("	# $2 = ($15 < $1)");
  print_("	cmpult $15,$1,$2"); 
  print_("	# if ($15 >= $1) goto " Lkf);
  print_("	beq $2," Lkf); 
  print_("	# $1 = ($15 < $30)");
  print_("	cmpult $15,$30,$1");
  print_("	# if ($15 < $30) goto " Lkf);
  print_("	bne $1," Lkf);
}

##
## free frame (SP = SP + X or SP = FP)
##

function action_free_frame_in_epilogue(\
				       sp)
{
  if (C_task == "asmpp") {
    sp = G_CP["SP"];
    if (L_TI["src"] == C_md_FP) {
      new_sp = G_CP["FP"] + L_TI["inc"];
    } else if (L_TI["src"] == C_md_SP) {
      new_sp = G_CP["SP"] + L_TI["inc"];
    } else {
      fatal("unknown free frame insn (either SP += X or SP = FP + X");
    }
    G_CP["SP"] = new_sp;
    if (C_dbg>0) dbg_print("SP " sp " ==> " new_sp);
    if (G_CP["postprocess"] && !G_CP["leaf"]) {
      ensure_check_free_frame();
    }
  } else if (C_task == "tblpp") {
  } else wrong_task();
  print_and_next_line(0);
}

##
## SP 
##

function action_inc_sp_in_epilogue(\
				   sp)
{
  if (C_task == "asmpp") {
    if (is_number(L_TI["offset"])) {
      o = 0 + L_TI["offset"];
    } else if ("const_reg" SUBSEP L_TI["offset"] in G_CP) {
      if (L_TI["sign"] == "+") {
	o = 0 + G_CP["const_reg",L_TI["offset"]];
      } else {
	o = 0 - G_CP["const_reg",L_TI["offset"]];
      }
    } else wrong("expected a constant offset to free frame");
    sp = G_CP["SP"];
    new_sp = sp + o;
    G_CP["SP"] = new_sp;
    if (C_dbg>0) dbg_print("SP " sp " ==> " new_sp);
    if (G_CP["postprocess"] && !G_CP["leaf"]) {
      ensure_check_free_frame();
    }
  } else if (C_task == "tblpp") {
  } else wrong_task();
  print_and_next_line(0);
}

##
## ensure check free frame sequence has been inserted
##

function ensure_check_free_frame()
{
  if (!("check_free_frame_inserted" in G_CP)) {
    if(C_dbg>0) dbg_print("----- check frame begin -----");
    n_e = G_CP["n_epilogues"];
    Lkf = make_compiler_symbol(G_CP["name"] C_proc_keep_frame_suffix n_e);
    check_free_frame(Lkf);
    if ("saved_reset_sp" in G_CP) {
      if(C_dbg>0) dbg_print("reset SP");
# FP is already destroyed, we cannot use it here
      print_("	" make_move(C_md_VSP, C_md_SP));
# if FP is supposed to be restored before free frame (as is the case in 
# most CPUs), FP has been saved in prologue, and FP has not been 
# restored up to this point, we restore it here
      if (!C_md_FP_is_restored_in_tail &&
	  ("saved_offset" SUBSEP C_md_FP in G_CP) && 
	  !("restored" SUBSEP C_md_FP in G_CP)) {
	print_restore_reg(G_CP["load_mnemonic",C_md_FP], 
			  G_CP["saved_offset",C_md_FP], C_md_FP, 0);
      }
    }
    G_CP["check_free_frame_inserted"] = 1;
    if(C_dbg>0) dbg_print("----- check frame end -----");
  }
}

##
## an instruction is found after a possible free frame insn has been scanned
## this instruction must be saved and copied after keep_frame version of the
## epilogue
##

function action_other_in_epilogue(\
				  n, l)
{
  l = $0;
  if (G_CP["postprocess"] && !G_CP["leaf"]) {
    if (index(l, C_md_FP) || index(l, C_md_SP) || index(l, C_md_VSP)) {
      syntax_error("insns that do not use " C_md_FP ", " C_md_SP ", or " C_md_VSP);
    }
    if ("check_free_frame_inserted" in G_CP) {
      n = G_CP["n_other_after_free_frame"];
      G_CP["n_other_after_free_frame"] = n + 1;
# save this instruction
      G_CP["other_after_free_frame",n] = l;
    }
  }
  print_and_next_line(0);
}

##
## return instruction
##

function action_return_in_epilogue()
{
  if (C_task == "asmpp") {
    if ("return_displacement" in G_CP) {
      if (G_CP["return_displacement"] != L_TI["disp"]) {
	wrong("inconsistent return displacement (previously " G_CP["return_displacement"] ")");
      }
    } else {
      G_CP["return_displacement"] = L_TI["disp"];
    }
    if (G_CP["postprocess"] && !G_CP["leaf"]) {
      ensure_check_free_frame();
    }
  } else if (C_task == "tblpp") {
  } else wrong_task();
  print_and_next_line(0);
}

##
## almost end epilogue (after return insn)
##

function action_aux_almost_end_epilogue(\
					Lkf, Lpe, pfpo, rao, n, r)
{
  if (C_task == "asmpp") {
    n_e = G_CP["n_epilogues"];
    G_CP["n_epilogues"] = n_e + 1;
    if (G_CP["postprocess"] && !G_CP["leaf"]) {
#      warn(G_CP["name"] " is postprocessed");
      check_insn_count();

# we count pure epilogues here, to tell the table generation procedure 
# (see gen_proc_decl etc.) whether we generated any pure epilogue or not. 
# the value of the counter is not very important, except for whether it is 
# zero or not. in particular, we still use n_epilouges to generate symbol 
# for pure epilogues (for humans readability). 
      n_pe = G_CP["n_pure_epilogues"];
      G_CP["n_pure_epilogues"] = n_pe + 1;
      Lkf = make_compiler_symbol(G_CP["name"] C_proc_keep_frame_suffix n_e);
      Lpe = make_compiler_symbol(G_CP["name"] C_proc_pure_epilogue_suffix n_e);
      fix_restore_base();
      if ("return_address_fp_offset" in G_CP) {
	rao = G_CP["return_address_fp_offset"] + G_CP["FP"] - G_CP["restore_base_offset"];
      } else {
	wrong("return address is not saved for non-leaf procedure " G_CP["name"]);
      }
      n = G_CP["n_saved_regs"];

# an epilogue tail that only restores FP 
      print_(Lkf ":");
      for (i = 0; i < G_CP["n_other_after_free_frame"]; i++) {
	if (C_dbg>0) dbg_print("instruction that is copied from epilogue");
	print_(G_CP["other_after_free_frame",i]); # newline?
      }
# restore regs that should be restored here
      for (i = n - 1; i >= 0; i--) {
	r = G_CP["saved_reg",i];
	if (!("restored" SUBSEP r in G_CP) && 
	    !index(C_md_unrestored_saved_regs, r)) {
	  print_restore_reg(G_CP["load_mnemonic",r], 
			    G_CP["saved_offset",r], r, 0);
	}
      }
      epilogue_tail_variation(1, G_CP["restore_base"], rao);
      
# pure epilogue 
# clear the flag "reset_restore_base" again to reset the restore base again
      delete G_CP["reset_restore_base"];
      print_(Lpe ":");
      for (i = n - 1; i >= 0; i--) {
	r = G_CP["saved_reg",i];
	if (!index(C_md_unrestored_saved_regs, r)) {
	  print_restore_reg(G_CP["load_mnemonic",r],
			    G_CP["saved_offset",r], r, 1);
	}
      }
      epilogue_tail_variation(0, G_CP["restore_base"], rao);
# put the end label
      print_(make_compiler_symbol(G_CP["name"] C_proc_end_suffix n_e) ":");
    } else {
#      warn(G_CP["name"] " is not postprocessed");
    }	
  } else if (C_task == "tblpp") {
  } else wrong_task();
}

function epilogue_tail_variation(fin, b, rao)
{
  if (C_cpu == "i386") {
    i386_epilogue_tail_variation(fin, b, rao);
  } else if (C_cpu == "sparc") {
    sparc_epilogue_tail_variation(fin, b, rao);
  } else if (C_cpu == "mips") {
    mips_epilogue_tail_variation(fin, b, rao);
  } else if (C_cpu == "alpha") {
    alpha_epilogue_tail_variation(fin, b, rao);
  } else wrong_port("cpu");
}

function i386_epilogue_tail_variation(fin, b, rao)
{
# load RA via VSP (%edx = VSP[o])
  print_("	movl " rao "(" b "),%edx");
  if (fin) {
# zero the RA slot (VSP[o] = 0)
    print_("	movl $0," rao "(" b ")");
# SP -= tls[8]
    print_("	subl 8(" C_md_TLS ")," C_md_SP);
  }
  print_("	jmp *%edx");
}

function sparc_epilogue_tail_variation(fin, b, rao)
{
  if (fin) {
    print_("	st %g0,[" b "+" rao "]");
  }
  if (G_CP["return_displacement"] == 8) {
    print_("	retl");
  } else if (G_CP["return_displacement"] == 12) {
    print_("	jmp %o7+12");
  } else wrong("wrong return displacement " G_CP["return_displacement"]);
  print_("	nop");
}

function mips_epilogue_tail_variation(fin, b, rao)
{
  if (fin) {
    print_("	sw	$0," rao "(" b ")");
  }
  print_("	j	$31");
}

function alpha_epilogue_tail_variation(fin, b, rao)
{
  if (fin) {
    print_("	stq	$31," rao "(" b ")");
  }
  print_("	ret	$31,($26),1");
}

##
## called after a procedure has been scanned.
## record various information about the procedure.
##

function action_aux_end_proc(\
			     n)
{
  if (C_task == "asmpp") {
    if ("idx" in G_CP) idx = G_CP["idx"];
    else fatal("proc info idx is not set for a procedure");
    if ("name" in G_CP) name = G_CP["name"];
    else fatal("proc info id undefined procedure ID = " idx);
    if (C_dbg>0) {
      dbg_print("---- end of procedure " name);
    }
    set_proc_info(idx, name, "name", "error");
    set_proc_info(idx, name, "return_address_fp_offset", -1);
    set_proc_info(idx, name, "parent_fp_fp_offset", -1);
    set_proc_info(idx, name, "base_SP-FP", "error");
    set_proc_info(idx, name, "sp_relative_size", "error");
    set_proc_info(idx, name, "max_sp_shrink_size", "error");
    set_proc_info(idx, name, "fork_block_start_idx", "error");
    set_proc_info(idx, name, "n_fork_blocks", "error");
    set_proc_info(idx, name, "invalid_call_start_idx", "error");
    set_proc_info(idx, name, "n_invalid_calls", "error");
    set_proc_info(idx, name, "return_displacement", 0);
    set_proc_info(idx, name, "n_epilogues", "error");
    set_proc_info(idx, name, "n_pure_epilogues", "error");
    set_proc_info(idx, name, "postprocess", "error");
    set_proc_info(idx, name, "leaf", "error");

    if (G_CP["leaf"] || G_CP["postprocess"] == 0) {
      G_unpostprocessed_procs[name] = 1;
    }
    
  } else if (C_task == "tblpp") {
  } else wrong_task();
}

##
## end of file
##

function gen_proc_decl(i,
		       name, prefix, start, epilogue, pure_epilogue)
{
  name = G_proc_info[i,"name"];
  prefix = C_md_compiler_symbol_prefix name;
  start = encode_name(prefix C_proc_start_suffix, 1);
  if (G_proc_info[i,"postprocess"] && G_proc_info[i,"n_epilogues"] > 0) {
    e_idx = G_proc_info[i,"n_epilogues"] - 1;
    epilogue = encode_name(prefix C_proc_epilogue_suffix e_idx, 1);
  } else {
    e_idx = -1;
    epilogue = 0;
  }
# !G_proc_info[i,"leaf"]
  if (epilogue && G_proc_info[i,"n_pure_epilogues"] > 0) {
    if (e_idx == -1) fatal("e_idx == -1");
    pure_epilogue = encode_name(prefix C_proc_pure_epilogue_suffix e_idx, 1);
  } else {
    pure_epilogue = 0;
  }
  printf("/* %s */\n", name) >> C_tab_c;
  printf("void %s();\n", start) >> C_tab_c;
  if (epilogue) {
    printf("void %s();\n", epilogue) >> C_tab_c;
  }
  if (pure_epilogue) {
    printf("void %s();\n", pure_epilogue) >> C_tab_c;
  }
}

function gen_proc_info(i,
		       name, prefix, start, epilogue, pure_epilogue, module,
		       e_idx)
{
  name = G_proc_info[i,"name"];
  prefix = C_md_compiler_symbol_prefix name;
  start = encode_name(prefix C_proc_start_suffix, 1);
  if (G_proc_info[i,"postprocess"] && G_proc_info[i,"n_epilogues"] > 0) {
    e_idx = G_proc_info[i,"n_epilogues"] - 1;
    epilogue = encode_name(prefix C_proc_epilogue_suffix e_idx, 1);
  } else {
    e_idx = -1;
    epilogue = "0";
  }
# !G_proc_info[i,"leaf"]
  if (epilogue != "0" && G_proc_info[i,"n_pure_epilogues"] > 0) {
    if (e_idx == -1) fatal("e_idx == -1");
    pure_epilogue = encode_name(prefix C_proc_pure_epilogue_suffix e_idx, 1);
  } else {
    pure_epilogue = "0";
  }
  module = encode_name(C_module, 0);
  
  printf(" {/* %s */\n", name) >> C_tab_c;
  printf("  /*begin=*/(uslong)%s,\n", start) >> C_tab_c;
  printf("  /*real_epilogue=*/(uslong)%s,\n", epilogue) >> C_tab_c;
  printf("  /*pure_epilogue=*/(uslong)%s,\n", pure_epilogue) >> C_tab_c;
  printf("  /*return address offset=*/%d,\n",
	 G_proc_info[i,"return_address_fp_offset"]) >> C_tab_c;
  printf("  /*parent fp offset=*/%d,\n",
	 G_proc_info[i,"parent_fp_fp_offset"]) >> C_tab_c;
  printf("  /*return displacement=*/%d,\n", 
	 G_proc_info[i,"return_displacement"]) >> C_tab_c;
  printf("  /*sp_relative_size=*/%d,\n", 
	 G_proc_info[i,"sp_relative_size"]) >> C_tab_c;
  printf("  /*max_sp_shrink_size=*/%d,\n",
	 G_proc_info[i,"max_sp_shrink_size"]) >> C_tab_c;
  printf("  /*base_SP-FP=*/%d,\n", 
	 G_proc_info[i,"base_SP-FP"]) >> C_tab_c;
  printf("  /*fork_point_info=*/%s%s + %d, %d,\n",
	 module, C_fork_point_info_suffix, 
	 G_proc_info[i,"fork_block_start_idx"], 
	 G_proc_info[i,"n_fork_blocks"]) >> C_tab_c;
  printf("  /*invalid_call_site_info=*/%s%s + %d, %d,\n",
	 module, C_invalid_call_site_info_suffix, 
	 G_proc_info[i,"invalid_call_start_idx"], 
	 G_proc_info[i,"n_invalid_calls"]) >> C_tab_c;
  printf("  /*next=*/0,\n")>> C_tab_c;
  printf("  /*begin_=*/%s,\n", start) >> C_tab_c;
  printf("  /*real_epilogue_=*/%s,\n", epilogue) >> C_tab_c;
  printf("  /*pure_epilogue_=*/%s,\n", pure_epilogue) >> C_tab_c;
  printf("  /*name=*/\"%s\"\n", name) >> C_tab_c;
  printf(" },\n") >> C_tab_c;
}

function gen_data_decl(i)
{
  name = G_data_info[i,"name"];
  printf("extern int %s[]; /* %s */\n", encode_name(name, 1), name) >> C_tab_c;
}

function gen_data_info(i)
{
  name = G_data_info[i,"name"];
  size = G_data_info[i,"size"];
  printf("  { /*addr=*/(uslong)%s, /*size=*/%d, /*name=*/\"%s\" },\n",
	 encode_name(name, 1), size, name) >> C_tab_c;
}

##
## BSS directive 
##

function action_directive_bss(\
			      n)
{
  n = G_n_data;
  G_n_data = n + 1;
  if(C_dbg>0) {
    dbg_print("BSS data " n " : " L_TI["name"] " (" L_TI["size"] " bytes)");
  }
  G_data_info[n,"name"] = L_TI["name"];
  G_data_info[n,"size"] = L_TI["size"];
  print_and_next_line(0);
}

##
## datum
##

function action_aux_begin_data()
{
  if(C_dbg>0) dbg_print("BEGIN data name = " G_last_C_label);
  delete G_DP;
  G_DP["name"] = G_last_C_label;
  G_DP["size"] = 0;
}

function action_datum(\
		      s, type, datum)
{
  if (C_task == "asmpp") {
    s = G_DP["size"];
    G_DP["size"] = s + L_TI["size"];
    if(C_dbg>0) dbg_print(s " bytes ===> " G_DP["size"] " bytes");
    print_and_next_line(0);
  } else if (C_task == "tblpp") {
    if (is_number(L_TI["datum"])) {
      datum = 0;
    } else {
      type = symbol_type(L_TI["datum"]);
      if (type == "C") {
	datum = decode_name(C_symbol_name(L_TI["datum"]));
      } else if (type == "compiler") {
	datum = unique_compiler_name(L_TI["datum"]);
      } else {
	datum = 0;
      }
    }

    if (datum) {
      if(C_dbg>0)dbg_print("decoded");
      print_and_next_line("	" L_TI["directive"] " " datum);
    } else {
      print_and_next_line(0);
    }
  } else wrong_task();
}

function action_aux_end_data(\
			     n)
{
  n = G_n_data;
  G_n_data = n + 1;
  if(C_dbg>0) dbg_print("END data " G_DP["name"]);
  G_data_info[n,"name"] = G_DP["name"];
  G_data_info[n,"size"] = G_DP["size"];
}

function n_unpostprocessed_procs(\
				 x)
{
  n = 0;
  for (x in G_unpostprocessed_procs) {
    n++;
  }
  return n;
}

function action_end_asm_file(\
			     i, prefix, nf, ni, module)
{
  if (C_task == "asmpp") {
    module = encode_name(C_module, 0);

    if (C_dbg>0) dbg_print(G_n_procs " procs " (G_n_procs - n_unpostprocessed_procs()) " postprocessed");
    
    printf("\n") > C_tab_c;
    printf("#define PROC_INFO_ONLY\n") >> C_tab_c;
    printf("#include \"st.h\"\n") >> C_tab_c;
    printf("\n") >> C_tab_c;
    
#
# ----- generate fork point declaration -----
# 
    printf("/* fork point declaration */\n") >> C_tab_c;
    for (i = 0; i < G_n_fork_blocks; i++) {
      printf("void %s%d(), %s%d();\n", 
	     encode_name(C_md_compiler_symbol_prefix C_fork_block_start_prefix, 1), i,
	     encode_name(C_md_compiler_symbol_prefix C_fork_block_end_prefix, 1), i) >> C_tab_c;
    }
#
# ----- generate fork point info -----
#
    printf("/* fork point info */\n") >> C_tab_c;
    printf("struct st_fork_point_info %s%s[] = {\n", 
	   module, C_fork_point_info_suffix) >> C_tab_c;
    for (i = 0; i < G_n_fork_blocks; i++) {
      printf("  { (uslong)%s%d, (uslong)%s%d },\n", 
	     encode_name(C_md_compiler_symbol_prefix C_fork_block_start_prefix, 1), i,
	     encode_name(C_md_compiler_symbol_prefix C_fork_block_end_prefix, 1), i) >> C_tab_c;
    }
    printf("  { 0, 0 } /* end marker */ };\n") >> C_tab_c;
    
#
# ----- generate invalid call site declaration -----
#
    printf("/* invalid call site declaration */\n") >> C_tab_c;
    for (i = 0; i < G_n_invalid_calls; i++) {
      printf("void %s%d(), %s%d();\n", 
	     encode_name(C_md_compiler_symbol_prefix C_invalid_call_start_prefix, 1), i,
	     encode_name(C_md_compiler_symbol_prefix C_invalid_call_end_prefix, 1), i) >> C_tab_c;
    }
    
#
# ----- generate invalid call site info -----
#
    printf("/* invalid call site info */\n") >> C_tab_c;
    printf("struct st_invalid_call_site_info %s%s[] = {\n", 
	   module, C_invalid_call_site_info_suffix) >> C_tab_c;
    for (i = 0; i < G_n_invalid_calls; i++) {
      printf("  { (uslong)%s%d, (uslong)%s%d },\n", 
	     encode_name(C_md_compiler_symbol_prefix C_invalid_call_start_prefix, 1), i,
	     encode_name(C_md_compiler_symbol_prefix C_invalid_call_end_prefix, 1), i) >> C_tab_c;
    }
    printf("  { 0, 0 } /* end marker */ };\n") >> C_tab_c;
    
#
# ----- generate procedure declaration -----
#
    printf("/* procedure declaration */\n") >> C_tab_c;
    for (i = 0; i < G_n_procs; i++) {
      gen_proc_decl(i);
    }
    
#
# ----- generate procedure information -----
#
    printf("/* procedure information */\n") >> C_tab_c;
    printf("struct st_proc_info %s%s[] = {\n", 
	   module, C_proc_info_suffix) >> C_tab_c;
    for (i = 0; i < G_n_procs; i++) {
      gen_proc_info(i);
    }
    printf("  { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 }\n") >> C_tab_c;
    printf("  /* end marker */ };\n") >> C_tab_c;
    
#
# ----- generate data declaration -----
#
    
    printf("/* data declaration (%d items) */\n", G_n_data) >> C_tab_c;
    for (i = 0; i < G_n_data; i++) {
      gen_data_decl(i);
    }
    
#
# ----- generate data info -----
#
    printf("/* data information */\n") >> C_tab_c;
    printf("struct st_data_info %s%s[] = {\n", 
	   module, C_data_info_suffix) >> C_tab_c;
    for (i = 0; i < G_n_data; i++) {
      gen_data_info(i);
    }
    printf("  { 0, 0, 0 } /* end marker */ };\n") >> C_tab_c;
  } else if (C_task == "tblpp") {

  } else wrong_task();
}
  
##
## move G_CP[x] to G_proc_info[idx,x]
##

function set_proc_info(idx, name, x, if_undefined)
{
  if (x in G_CP) {
    if (C_dbg>0) {
      dbg_print("  " x " of " name " = " G_CP[x]);
    }
    G_proc_info[idx,x] = G_CP[x];
  } else if (if_undefined == "error") {
    fatal("proc info " x " undefined for procedure " name);
  } else if (if_undefined == "undefined") {
# do nothing
  } else {
    G_proc_info[idx,x] = if_undefined;
  }
}

##
## ---------- some trivial machine-dependent procedures ----------
##

##
## make_offset_pointer(b, s, o, d):
## make an instruction that does "D = S + o" (when s == "+") or "D = S - o"
## (when s == "-")
##

function make_offset_pointer(b, s, o, d,
			     r)
{
  if (C_cpu == "i386") {
    r = i386_make_offset_pointer(b, s, o, d);
  } else if (C_cpu == "sparc") {
    r = sparc_make_offset_pointer(b, s, o, d);
  } else if (C_cpu == "mips") {
    r = mips_make_offset_pointer(b, s, o, d);
  } else if (C_cpu == "alpha") {
    r = alpha_make_offset_pointer(b, s, o, d);
  } else wrong_port("cpu");
  return r;
}

function i386_make_offset_pointer(b, s, o, d)
{
# reset SP looks like "leal -??(%ebp),%esp". we simply substitute 
# the virtual SP for %esp
  if (is_number(o)) {
    if (o == 0) {
      return "movl " b "," d;
    } else if (s == "+") {
      return "leal " o "(" b ")," d;
    } else wrong("i386_make_offset_pointer(" b "," s "," o "," d "): invalid arg");
  } else {
    wrong("i386_make_offset_pointer(" b "," s "," o "," d "): invalid arg");
  }
}

function sparc_make_offset_pointer(b, s, o, d)
{
  if (s == "+") {
    return "add " b "," o "," d;
  } else if (s == "-") {
    return "sub " b "," o "," d;
  } else wrong("sparc_make_offset_pointer(" b "," s "," o "," d "): wrong sign");
}

function mips_make_offset_pointer(b, s, o, d)
{
  if (o == 0) {
    return "move	" d "," b;
  } else if (s == "+") {
    return "addu	" d "," o "," b;
  } else if (s == "-") {
    return "subu	" d "," o "," b;
  } else wrong("mips_make_offset_pointer(" b "," s "," o "," d "): wrong sign");
}

function alpha_make_offset_pointer(b, s, o, d)
{
  if (o == 0) {
    return "bis " b "," b "," d;
  } else wrong("alpha_make_offset_pointer(" o "): non-zero offset");
}

##
## make_load(ld, b, o, d):
## make an instruction that does "d = b[o]". use ld for mnemonic
##

function make_load(ld, b, o, d,
		   r)
{
  if (C_cpu == "i386") {
    r = i386_make_load(ld, b, o, d);
  } else if (C_cpu == "sparc") {
    r = sparc_make_load(ld, b, o, d);
  } else if (C_cpu == "mips") {
    r = mips_make_load(ld, b, o, d);
  } else if (C_cpu == "alpha") {
    r = alpha_make_load(ld, b, o, d);
  } else wrong_port("cpu");
  return r;
}

function i386_make_load(ld, b, o, d)
{
  return ld " " o "(" b ")," d;
}

function sparc_make_load(ld, b, o, d,
			 sign)
{
  if (o >= 0) sign = "+";
  else sign = "";
  return ld " [" b sign o "]," d;
}

function mips_make_load(ld, b, o, d)
{
  return ld "	" d "," o "(" b ")";
}

function alpha_make_load(ld, b, o, d)
{
  return ld " " d "," o "(" b ")";
}

##
## make_move 
## this must return an instruction that does "D = S", where D and S are 
## integer registers
##

function make_move(s, d)
{
  if (C_cpu == "i386") {
    r = i386_make_move(s, d);
  } else if (C_cpu == "sparc") {
    r = sparc_make_move(s, d);
  } else if (C_cpu == "mips") {
    r = mips_make_move(s, d);
  } else if (C_cpu == "alpha") {
    r = alpha_make_move(s, d);
  } else wrong_port("cpu");
  return r;
}

function i386_make_move(s, d)
{
  return "movl " s "," d;
}

function sparc_make_move(s, d)
{
  return "mov " s "," d;
}

function mips_make_move(s, d)
{
  return "move	" d "," s;
}

function alpha_make_move(s, d)
{
  return "bis " s "," s "," d;
}

##
## d = v
##

function make_const(v, d)
{
  if (C_cpu == "i386") {
    r = i386_make_const(v, d);
  } else if (C_cpu == "sparc") {
    r = sparc_make_const(v, d);
  } else if (C_cpu == "mips") {
    r = mips_make_const(v, d);
  } else if (C_cpu == "alpha") {
    r = alpha_make_const(v, d);
  } else wrong_port("cpu");
  return r;
}

function i386_make_const(v, d)
{
  return "movl " v "," d;
}

function sparc_make_const(v, d)
{
  return "set " v "," d;
}

function mips_make_const(v, d)
{
  return "li	" d ",0x" write_hex(v);
}

function alpha_make_const(v, d)
{
  return "addq $31," v "," d;
}

##
## if (X == 0) print current line;
## if (X != 0) print X;
##

function print_(x)
{
  if (C_dbg == 0) {
    if (x == 0) print;
    else print x;
  } else {
    if (x == 0) x = $0;
    printf("%s PRT[%4d]: %s\n", C_md_comment_string, FNR, x);
  }
}

##
## if (X == 0) comment out current line;
## if (X != 0) comment out X;
##

function comment_(x)
{
  if (x == 0) x = $0;
  if (C_dbg == 0) {
    print " " C_md_comment_string " " x;
  } else {
    printf("%s CMT[%4d]: %s %s\n", 
	   C_md_comment_string, FNR, C_md_comment_string, x);
  }
}

##
## read next line and zero L_tok to indicate that the line has not been
## tokenized. the scanner must perform peek_token to validate L_tok.
## as an exception, L_tok is set to "tk_eof" if EOF is reached.
##

function next_line()
{
  if (getline) {
# indicate that we have not yet tokenized this line
    L_tok = 0;			
  } else {
# indicate that we have reached EOF
    L_tok = "tk_eof";
  }
}

##
## make sure the current line has been tokenized and is interesting.
##

function peek_token()
{
  if (L_tok == 0) {		# this line has not been tokenized
    while (1) {
      L_tok = tokenize_line();
      if (L_tok) break;
      print_(0);
      if (getline == 0) {
	L_tok = "tk_eof";
	break;
      } 
    }
  }
}

function peek_token_commenting()
{
  if (L_tok == 0) {		# this line has not been tokenized
    while (1) {
      L_tok = tokenize_line();
      if (L_tok) break;
      comment_(0);
      if (getline == 0) {
	L_tok = "tk_eof";
	break;
      } 
    }
  }
}

function reset_insn_count()
{
  G_CP["n_insns"] = 0;
}

function tick_insn_count()
{
  G_CP["n_insns"]++;
}

function check_insn_count()
{
  if (C_check_insn_count) {
    if (G_CP["n_insns"] < 15) {
      warn(G_CP["name"] " has only " G_CP["n_insns"] " insns, yet postprocessed");
    }
  }
}

##
## print_(x) and then repeat reading lines until we reach an interesting line
##

function print_and_next_line(x)
{
  print_(x);
  next_line();
}

##
## comment_(x) and then repeat reading lines until we reach an interesting line
##

function comment_and_next_line(x)
{
  comment_(x);
  next_line();
}

##
## return 1 if tok is in FIRST set of non-terminal NT
##

function in_FIRST(tok, NT)
{
  if (NT in C_FIRST) {
# space serves as the word separator
    if (index(C_FIRST[NT], tok " ")) {
      return 1;
    } else {
      return 0;
    }
  } else {
    fatal("unknown non-terminal " NT);
  }
} 

##
## machine-dependent line scanner
##

function i386_tokenize_line(l,
			    name, type, f, bo)
{
  if (match(l, /^.+:$/)) {
    return tokenize_label(substr(l, 1, RLENGTH - 1));
  } else if (match(l, /^\./)) {
# some kind of directives
    L_TI["directive"] = L_tokens[1];
    if (match(l, /^\.file.*\".*\"$/)) {
# .file "foo.c"
      return "tk_directive_file";
    } else if (match(l, /^\.string[[:space:]]+\".*\"$/)) {
# .string "foo"
      return "tk_string";
    } else if (match(l, /^\.comm/)) {
# .comm a,400,4 (BSS data declaration)
      if (C_os == "solaris" || C_os == "linux") {
	if (match(l, /^\.comm[[:space:]]+[^,]+,[[:digit:]]+,[[:digit:]]+/)) {
	  L_TI["name"] = L_tokens[2,1];
	  L_TI["size"] = L_tokens[2,2];
	} else wrong("unexpected bss syntax " l);
      } else if (C_os == "winnt") {
#  .comm	_bss_iro, 6400	 # 6400
	if (match(l, /^\.comm[[:space:]]+[^,]+,[[:space:]]+[[:digit:]]+/)) {
	  L_TI["name"] = L_tokens[2,1];
	  L_TI["size"] = L_tokens[3];
	} else wrong("unexpected bss syntax " l);
      } else wrong_port("os 5");
      return "tk_directive_bss";
    } else {
# others
      if (C_os == "solaris" || C_os == "linux") {
	if (match(l, /^\.zero[[:space:]]/)) {
	  L_TI["datum"] = 0;
	  L_TI["size"] = 0 + L_tokens[2];
	  return "tk_datum";
	} else if (match(l, /^\.(byte|value|long)[[:space:]]/)) {
	  L_TI["datum"] = L_tokens[2];
	  L_TI["size"] = C_md_data_size[L_tokens[1]];
	  return "tk_datum";
	} else {
	  return 0;		# uninteresting line
	}
      } else if (C_os == "winnt") {
	if (match(l, /^\.space[[:space:]]/)) {
	  L_TI["datum"] = 0;
	  L_TI["size"] = 0 + L_tokens[2];
	  return "tk_datum";
	} else if (match(l, /^\.(byte|word|long)[[:space:]]/)) {
	  L_TI["datum"] = L_tokens[2];
	  L_TI["size"] = C_md_data_size[L_tokens[1]];
	  return "tk_datum";
	} else {
	  return 0;		# uninteresting line
	}
      } else wrong_port("os 6");
    } 
# end of directive processing
  } else if (index(l, C_md_comment_string)) {
# comment
    if (match(l, /\/ REAL INSN BEGINS/)) {
      return "tk_comment_real_insn_begins";
    } else if (match(l, /\/ REAL INSN ENDS/)) {
      return "tk_comment_real_insn_ends";
    } else return 0;
  } else {


# instructions
    if (match(l, /^popl %.../)) {
# popl xxx (either register restore or some float conversion ops?)
      L_TI["base"] = C_md_SP;
      L_TI["offset"] = 0;
      L_TI["dest"] = L_tokens[2];
      L_TI["op_size"] = 4;
      L_TI["load_mnemonic"] = "movl";
      L_TI["post_inc_sp"] = 4;
      return "tk_insn_load";
    } else if (match(l, /^pushl %.../)) {
# push xxx (either register save or argument store)
      L_TI["base"] = C_md_SP;
      L_TI["offset"] = 0;
      L_TI["src"] = L_tokens[2];
      L_TI["op_size"] = 4;
      L_TI["load_mnemonic"] = "movl";
      L_TI["pre_inc_sp"] = -4;
      return "tk_insn_store";
    } else if (match(l, /^call/)) {
# special functions such as call __st_fork_block_start
      f = L_tokens[2];
      type = symbol_type(f);
      L_TI["opc"] = "call";
      L_TI["target"] = f;
      if (type == "C" && 
	  index(C_special_procedures, C_symbol_name(f) " ")) {
	return "tk_insn_special_proc";
      } else {
	return "tk_insn_call";
      }
    } else if (match(l, /^ret/)) {
      L_TI["disp"] = 0;
      return "tk_insn_return";
    } else if (match(l, /^(jmp|jn?[ablg]?e?|jn?s)/)) {
# jumps (indicate the end of prologue)
      L_TI["opc"] = L_tokens[1];
      L_TI["target"] = L_tokens[2];
      return "tk_insn_jmp";
    } else if (match(l, /^(add|sub)l \$[[:digit:]]+,%esp$/)) {
      L_TI["sign"] = (L_tokens[1] == "addl" ? "+" : "-");
      match(L_tokens[2,1], /[[:digit:]]+/);
      L_TI["offset"] = 0 + substr(L_tokens[2,1], RSTART, RLENGTH);
      return "tk_insn_inc_sp";
    } else if (match(l, /^movl %esp,%ebp$/)) {
      L_TI["sign"] = "+";
      L_TI["offset"] = "0";
      return "tk_insn_setup_fp";
    } else if (match(l, /^leal -?[[:digit:]]+\(%ebp\),%esp/)) {
# leal -20(%ebp),%esp (SP reset)
      split(L_tokens[2,1], bo, /\(|\)/);
      L_TI["sign"] = "+";	# we know offset is a constant
      L_TI["offset"] = bo[1];
      L_TI["base"] = "%ebp";
      L_TI["dest"] = "%esp";
      return "tk_insn_reset_sp";
    } else if (match(l, /^movl %ebp,%esp$/)) {
# movl %ebp,%esp
      L_TI["inc"] = 0;
      L_TI["src"] = "%ebp";
      return "tk_insn_free_frame";
    } else if (match(l, /^leave$/)) {
# leave, which has two effects (SP, FP) = (FP + 4, FP[0])
# %esp = %ebp + 4
      L_TI["src"] = "%ebp";
      L_TI["inc"] = 4;
# %ebp = 0(%ebp)
      L_TI["base"] = "%ebp";
      L_TI["offset"] = 0;
      L_TI["dest"] = "%ebp";
      return "tk_insn_free_frame+sp_load";
    } else if (NF == 0 || match(l, /^\./)) {
# blank line or directive
      return 0;			# uninteresting line
    } else {
# other instructions
      return "tk_insn_other";
    }
  }
}

function sparc_tokenize_line(l,
			     name, type, f, a, bo)
{
  if (match(l, /^.+:$/)) {
    return tokenize_label(substr(l, 1, RLENGTH - 1));
  } else if (match(l, /^\./)) {
# some kind of directives
    L_TI["directive"] = L_flat_tokens[1];
    if (match(l, /^\.file.*\".*\"$/)) {
# .file "foo.c"
      return "tk_directive_file";
    } else if (match(l, /^\.asciz[[:space:]]+\".*\"$/)) {
# .asciz "foo"
      return "tk_string";
    } else if (match(l, /^\.common/)) {
# .comm a,400,4 (BSS data declaration)
      if (C_os == "solaris") {
	if (match(l, /^\.common[[:space:]]+[^,]+,[[:digit:]]+,[[:digit:]]+/)) {
	  L_TI["name"] = L_flat_tokens[2];
	  L_TI["size"] = L_flat_tokens[3];
	} else wrong("unexpected bss syntax " l);
      } else if (C_os == "linux") {
	wrong_port("os 7");
      } else wrong_port("os 8");
      return "tk_directive_bss";
    } else {
# others
      if (C_os == "solaris") {
	if (match(l, /^\.skip[[:space:]]/)) {
	  L_TI["datum"] = 0;
	  L_TI["size"] = 0 + L_flat_tokens[2];
	  return "tk_datum";
	} else if (match(l, /^\.(byte|(ua)?(half|word))[[:space:]]/)) {
	  L_TI["datum"] = L_flat_tokens[2];
	  L_TI["size"] = C_md_data_size[L_tokens[1]];
	  return "tk_datum";
	} else {
	  return 0;		# uninteresting line
	}
      } else if (C_os == "linux") {
	wrong_port("os 9");
      } else wrong_port("os 10");
    } 
# end of directive processing
  } else if (index(l, C_md_comment_string) == 1) {
# comment
    if (match(l, /! REAL INSN BEGINS/)) {
      return "tk_comment_real_insn_begins";
    } else if (match(l, /! REAL INSN ENDS/)) {
      return "tk_comment_real_insn_ends";
    } else if (l == "!#PROLOGUE# 1") {
      return "tk_hint_prologue_end";
    } else if (l == "!#PROLOGUE# 0") {
      return "tk_hint_prologue_begin";
    } else if (l == "!#EPILOGUE#") {
      return "tk_hint_epilogue_begin";
    } else return 0;
  } else {
# instructions
# Sep 13, 99. handle mnemonic followed by a tab (not space)
# and commas followed by spaces
    if (match(l, /^set[[:blank:]][[:digit:]]+,[[:blank:]]*%../)) {
# set ????,%??
      L_TI["op"] = "+";
      L_TI["dest"] = L_flat_tokens[3];
      L_TI["src",1] = 0;
      L_TI["src",2] = 0 + L_flat_tokens[2];
      return "tk_insn_arith";
    } else if (match(l, /^sethi[[:blank:]]%hi\(-?[[:digit:]]+\),[[:blank:]]+%../)) {
# sethi %hi(????),%??
      L_TI["op"] = "+";
      L_TI["dest"] = L_flat_tokens[3];
      split(L_flat_tokens[2], a, /\(|\)/); # %hi(??) ==> %hi ??
      if (!is_number(a[2])) {
	wrong("wrong split " L_flat_tokens[2] " ==> " a[1] " " a[2]);
      }
      L_TI["src",1] = 0;
      L_TI["src",2] = a[2] - (a[2] % 4096);
      return "tk_insn_arith";
    } else if (match(l, /^ldd?[[:blank:]]\[%..\+[[:digit:]]+\],[[:blank:]]*%../)) {
# ld [%sp+??],%??
      split(L_flat_tokens[2], a, /\[|\]|\+/); # [%sp+xx] ==> "" %sp xx
      if (!is_number(a[3])) {
	fatal("wrong split " L_flat_tokens[2] " ==> " a[1] " " a[2] " " a[3]);
      }
      L_TI["dest"] = L_flat_tokens[3];
      L_TI["base"] = a[2];
      L_TI["offset"] = 0 + a[3];
      if (match(l, /^ldd/)) {
	L_TI["op_size"] = 8;
	L_TI["load_mnemonic"] = "ldd";
      } else {
	L_TI["op_size"] = 4;
	L_TI["load_mnemonic"] = "ld";
      }
      return "tk_insn_load";
    } else if (match(l, /^std?[[:blank:]]%..,[[:blank:]]*\[%..\+[[:digit:]]+\]/)) {
# we currently assume that the offset of any interesting store is constant
# st %??,[%??+??]
      split(L_flat_tokens[3], a, /\[|\]|\+/);
      if (!is_number(a[3])) {
	fatal("wrong split " L_flat_tokens[2] " ==> " a[1] " " a[2] " " a[3]);
      }
      L_TI["base"] = a[2];
      L_TI["offset"] = 0 + a[3];
      L_TI["src"] = L_flat_tokens[2];
      if (match(l, /^std/)) {
	L_TI["op_size"] = 8;
	L_TI["load_mnemonic"] = "ldd";
      } else {
	L_TI["op_size"] = 4;
	L_TI["load_mnemonic"] = "ld";
      }
      return "tk_insn_store";
    } else if (match(l, /^call/)) {
# special functions such as call __st_fork_block_start
      f = L_flat_tokens[2];
      type = symbol_type(f);
      L_TI["opc"] = "call";
      L_TI["target"] = f;
      if (type == "C" && 
	  index(C_special_procedures, C_symbol_name(f) " ")) {
	return "tk_insn_special_proc";
      } else {
	return "tk_insn_call";
      }
    } else if (match(l, /^retl/)) {
      L_TI["disp"] = 8;
      return "tk_insn_return";
    } else if (match(l, /^jmp[[:blank:]]%o7\+12/)) {
      L_TI["disp"] = 12;
      return "tk_insn_return";
    } else if (match(l, /^(jmp|b(g|l|n)?e?)/)) {
# jumps (indicate the end of prologue)
      L_TI["opc"] = L_flat_tokens[1];
      L_TI["target"] = L_flat_tokens[2];
      return "tk_insn_jmp";
    } else if (match(l, /^(add|sub)[[:blank:]]%sp,[[:blank:]]*[^,]+,[[:blank:]]*%sp/)) {
# add %sp,??,%sp
# sub %sp,??,%sp
      L_TI["sign"] = (L_flat_tokens[1] == "add" ? "+" : "-");
      L_TI["offset"] = L_flat_tokens[3];
      return "tk_insn_inc_sp";
    } else if (match(l, /^(add|sub)[[:blank:]]%sp,[[:blank:]]*[^,]+,[[:blank:]]*%i7/)) {
# add %sp,??,%i7
# sub %sp,??,%i7
      L_TI["sign"] = (L_flat_tokens[1] == "add" ? "+" : "-");
      L_TI["offset"] = L_flat_tokens[3];
      return "tk_insn_setup_fp";
    } else if (match(l, /^sub[[:blank:]]%i7,[[:blank:]]*[^,]+,[[:blank:]]*%sp/)) {
# sub %i7,??,%sp
      L_TI["sign"] = "-";
      L_TI["offset"] = L_flat_tokens[3];
      L_TI["base"] = "%i7";
      L_TI["dest"] = "%sp";
      return "tk_insn_reset_sp";
    } else if (NF == 0 || match(l, /^\./)) {
# blank line or directive
      return 0;			# uninteresting line
    } else {
# other instructions
      return "tk_insn_other";
    }
  }
}

function mips_tokenize_line(l,
			    name, type, f, a, bo)
{
  if (index(l, C_md_comment_string) == 1) {
# comment
    if (match(l, /# REAL INSN BEGINS/)) {
      return "tk_comment_real_insn_begins";
    } else if (match(l, /# REAL INSN ENDS/)) {
      return "tk_comment_real_insn_ends";
    } else return 0;
  } else if (match(l, /^.+:$/)) {
    return tokenize_label(substr(l, 1, RLENGTH - 1));
  } else if (match(l, /^\./)) {
# some kind of directives
    L_TI["directive"] = L_tokens[1];
    if (match(l, /^\.file.*\".*\"$/)) {
# .file "foo.c"
      return "tk_directive_file";
    } else if (match(l, /^\.comm/)) {
# .comm a,400
      if (C_os == "irix") {
	if (match(l, /^\.comm[[:space:]]+[^,]+,[[:digit:]]+/)) {
	  L_TI["name"] = L_tokens[2,1];
	  L_TI["size"] = L_tokens[2,2];
	} else wrong("unexpected bss syntax " l);
      } else wrong_port("os");
      return "tk_directive_bss";
    } else if (match(l, /^\.set[[:space:]]+reorder/)) {
      return "tk_directive_reorder";
    } else if (match(l, /^\.set[[:space:]]+noreorder/)) {
      return "tk_directive_noreorder";
    } else {
# others
      if (C_os == "irix") {
	if (match(l, /^\.space[[:space:]]/)) {
	  L_TI["datum"] = 0;
	  L_TI["size"] = 0 + L_tokens[2];
	  return "tk_datum";
	} else if (match(l, /^\.(byte|half|word)[[:space:]]/)) {
	  L_TI["datum"] = L_tokens[2];
	  L_TI["size"] = C_md_data_size[L_tokens[1]];
	  return "tk_datum";
	} else {
	  return 0;		# uninteresting line
	}
      } else wrong_port("os");
    } 
# end of directive processing
  } else {
# instructions
    if (match(l, /^li[[:space:]]+\$..?,0x[[:digit:]]+/)) {
      L_TI["op"] = "+";
      L_TI["dest"] = L_tokens[2,1];
      L_TI["src",1] = 0;
      L_TI["src",2] = read_hex(substr(L_tokens[2,2], 3));
      return "tk_insn_arith";
    } else if (match(l, /^l(w|d|\.d)[[:space:]]+\$f?..?,-?[[:digit:]]+\(\$..?\)/)) {
# lw $??,??($??)
      split(L_tokens[2,2], a, /\(|\)/);	# ??($??) ==> ?? ??
      if (!is_number(a[1])) {
	fatal("wrong split " L_tokens[2,2] " ==> " a[1] " " a[2]);
      }
      L_TI["dest"] = L_tokens[2,1];
      L_TI["base"] = a[2];
      L_TI["offset"] = 0 + a[1];
      if (match(l, /^lw/)) {
	L_TI["op_size"] = 4;
	L_TI["load_mnemonic"] = "lw";
      } else if (match(l, /^ld/)) {
	L_TI["op_size"] = 8;
	L_TI["load_mnemonic"] = "ld";
      } else if (match(l, /^l\.d/)) {
	L_TI["op_size"] = 8;
	L_TI["load_mnemonic"] = "l.d";
      } else fatal("unknown load mnemonic " L_tokens[1]);
      return "tk_insn_load";
    } else if (match(l, /^s(w|d|\.d)[[:space:]]+\$f?..?,-?[[:digit:]]+\(\$..?\)/)) {
# sw $??,??($??) 
      split(L_tokens[2,2], a, /\(|\)/);	# ??($??) ==> ??  ??
      if (!is_number(a[1])) {
	fatal("wrong split " L_tokens[2,2] " ==> " a[1] " " a[2]);
      }
      L_TI["base"] = a[2];
      L_TI["offset"] = 0 + a[1];
      L_TI["src"] = L_tokens[2,1];
      if (match(l, /^sw/)) {
	L_TI["op_size"] = 4;
	L_TI["load_mnemonic"] = "lw";
      } else if (match(l, /^sd/)) {
	L_TI["op_size"] = 8;
	L_TI["load_mnemonic"] = "ld";
      } else if (match(l, /^s\.d/)) {
	L_TI["op_size"] = 8;
	L_TI["load_mnemonic"] = "l.d";
      } else fatal("unknown store mnemonic " L_tokens[1]);
      return "tk_insn_store";
    } else if (match(l, /^jal/)) {
# special functions such as call __st_fork_block_start
      f = L_tokens[2];
      type = symbol_type(f);
      L_TI["opc"] = "call";
      L_TI["target"] = f;
      if (type == "C" && 
	  index(C_special_procedures, C_symbol_name(f) " ")) {
	return "tk_insn_special_proc";
      } else {
	return "tk_insn_call";
      }
    } else if (match(l, /^j[[:space:]]+\$31/)) {
      L_TI["disp"] = 0;
      return "tk_insn_return";
    } else if (match(l, /^(b(g|l)(e|t)z)/) || match(l, /^b(ne|eq)/)) {
# jumps (indicate the end of prologue)
      L_TI["opc"] = L_tokens[1];
      L_TI["target"] = L_tokens[2];
      return "tk_insn_jmp";
    } else if (match(l, /^(add|sub)u[[:space:]]+\$sp,\$sp,+/)) {
# addu $sp,$sp,40
      L_TI["sign"] = (L_tokens[1] == "addu" ? "+" : "-");
      L_TI["offset"] = L_tokens[2,3];
      return "tk_insn_inc_sp";
    } else if (match(l, /^move[[:space:]]+\$fp,\$sp/)) {
# move $fp,$sp
      L_TI["sign"] = "+";
      L_TI["offset"] = "0";
      return "tk_insn_setup_fp";
    } else if (match(l, /^addu[[:space:]]+\$..?,\$..?,\$sp/)) {
      L_TI["sign"] = "+";
      L_TI["dest"] = L_tokens[2,1];
      L_TI["offset"] = L_tokens[2,2];
      return "tk_insn_offset_sp";
    } else if (match(l, /^move[[:space:]]+\$sp,\$fp/)) {
# move $sp,$fp
      L_TI["sign"] = "+";
      L_TI["offset"] = "0";
      L_TI["base"] = "$fp";
      L_TI["dest"] = "$sp";
      return "tk_insn_reset_sp";
    } else if (NF == 0 || match(l, /^\./)) {
# blank line or directive
      return 0;			# uninteresting line
    } else {
# other instructions
      return "tk_insn_other";
    }
  }
}

function alpha_tokenize_line(l,
			     a)
			     
{
  if (index(l, C_md_comment_string) == 1) {
# comment
    if (match(l, /# REAL INSN BEGINS/)) {
      return "tk_comment_real_insn_begins";
    } else if (match(l, /# REAL INSN ENDS/)) {
      return "tk_comment_real_insn_ends";
    } else return 0;
  } else if (match(l, /^.+:$/)) {
    return tokenize_label(substr(l, 1, RLENGTH - 1));
  } else if (match(l, /^\./)) {
# some kind of directives
    L_TI["directive"] = L_tokens[1];
    if (match(l, /^\.file.*\".*\"$/)) {
# .file "foo.c"
      return "tk_directive_file";
    } else if (match(l, /^\.ascii[[:space:]]+\".*\"$/)) {
# .ascii "foo"
      return "tk_string";
    } else if (match(l, /^\.comm/)) {
# .comm a,400
      if (C_os == "osf1" || C_os == "linux") {
	if (match(l, /^\.comm[[:space:]]+[^,]+,[[:digit:]]+/)) {
	  L_TI["name"] = L_tokens[2,1];
	  L_TI["size"] = L_tokens[2,2];
	} else wrong("unexpected bss syntax " l);
      } else wrong_port("os");
      return "tk_directive_bss";
    } else {
# others
      if (C_os == "osf1" || C_os == "linux") {
	if (match(l, /^\.space[[:space:]]/) || match(l, /^\.zero[[:space:]]/)) {
	  L_TI["datum"] = 0;
	  L_TI["size"] = 0 + L_tokens[2];
	  return "tk_datum";
	} else if (match(l, /^\.(byte|word|long|quad|t_floating) /)) {
	  L_TI["datum"] = L_tokens[2];
	  L_TI["size"] = C_md_data_size[L_tokens[1]];
	  return "tk_datum";
	} else {
	  return 0;		# uninteresting line
	}
      } else wrong_port("os");
    } 
# end of directive processing
  } else {
# instructions
    if (match(l, /^(add|sub)q [^,]+,[^,]+,[^,]+/)) {
      if (L_tokens[2,3] == "$30") {
	L_TI["sign"] = (L_tokens[1] == "addq" ? "+" : "-");
	if (L_tokens[2,1] == "$30") {
	  L_TI["offset"] = L_tokens[2,2];
	} else if (L_tokens[2,2] == "$30") {
	  L_TI["offset"] = L_tokens[2,1];
	} else syntax_error("one of the src is $30");
	return "tk_insn_inc_sp";
      } else {
	L_TI["op"] = (L_tokens[1] == "addq" ? "+" : "-");
	L_TI["dest"] = L_tokens[2,3];
	L_TI["src",1] = (L_tokens[2,1] == "$31" ? 0 : L_tokens[2,1]);
	L_TI["src",2] = (L_tokens[2,2] == "$31" ? 0 : L_tokens[2,2]);
	return "tk_insn_arith";
      }
    } else if (match(l, /^lda \$..?,-?[[:digit:]]+\(\$..?\)/)) {
      split(L_tokens[2,2], a, /\(|\)/);
      if (!is_number(a[1])) {
	wrong("wrong split " L_tokens[2,2] " => " a[1] " " a[2]);
      }
      if (L_tokens[2,1] == "$30" && a[2] == "$30") {
	L_TI["sign"] = "+";
	L_TI["offset"] = a[1];
	return "tk_insn_inc_sp";
      } else {
	L_TI["op"] = "+";
	L_TI["dest"] = L_tokens[2,1];
	L_TI["src",1] = a[1];
	L_TI["src",2] = (a[2] == "$31" ? 0 : a[2]);
	return "tk_insn_arith";
      }
    } else if (match(l, /^ldah \$..?,-?[[:digit:]]+\(\$..?\)/)) {
# z = x * 65536 + y
      L_TI["op"] = "+";
      L_TI["dest"] = L_tokens[2,1];
      split(L_tokens[2,2], a, /\(|\)/);
      if (!is_number(a[1])) {
	wrong("wrong split " L_tokens[2,2] " ==> " a[1] " " a[2]);
      }
      L_TI["src",1] = a[1] * 65536;
      L_TI["src",2] = (a[2] == "$31" ? 0 : a[2]);
      return "tk_insn_arith";
    } else if (match(l, /^ld(q|t) \$f?..?,-?[[:digit:]]+\(\$..?\)/)) {
# ldq $??,??($??)
      split(L_tokens[2,2], a, /\(|\)/);	# ??($??) ==> ?? ??
      if (!is_number(a[1])) {
	fatal("wrong split " L_tokens[2,2] " ==> " a[1] " " a[2]);
      }
      L_TI["dest"] = L_tokens[2,1];
      L_TI["base"] = a[2];
      L_TI["offset"] = 0 + a[1];
      if (match(l, /^ldq/)) {
	L_TI["op_size"] = 8;
	L_TI["load_mnemonic"] = "ldq";
      } else if (match(l, /^ldt/)) {
	L_TI["op_size"] = 8;
	L_TI["load_mnemonic"] = "ldt";
      } else fatal("unknown load mnemonic " L_tokens[1]);
      return "tk_insn_load";
    } else if (match(l, /^st(q|t) \$..?,-?[[:digit:]]+\(\$..?\)/)) {
# stq $??,??($??) 
      split(L_tokens[2,2], a, /\(|\)/);	# ??($??) ==> ??  ??
      if (!is_number(a[1])) {
	fatal("wrong split " L_tokens[2,2] " ==> " a[1] " " a[2]);
      }
      L_TI["base"] = a[2];
      L_TI["offset"] = 0 + a[1];
      L_TI["src"] = L_tokens[2,1];
      if (match(l, /^stq/)) {
	L_TI["op_size"] = 8;
	L_TI["load_mnemonic"] = "ldq";
      } else if (match(l, /^stt/)) {
	L_TI["op_size"] = 8;
	L_TI["load_mnemonic"] = "ldt";
      } else fatal("unknown store mnemonic " L_tokens[1]);
      return "tk_insn_store";
    } else if (match(l, /^[bj]sr \$26,/)) {
# special functions such as call __st_fork_block_start
      f = L_tokens[2,2];
      type = symbol_type(f);
      L_TI["opc"] = "call";
      L_TI["target"] = f;
      if (type == "C" && 
	  index(C_special_procedures, C_symbol_name(f) " ")) {
	return "tk_insn_special_proc";
      } else {
	return "tk_insn_call";
      }
    } else if (match(l, /^ret \$31,\(\$26\),1/)) {
      L_TI["disp"] = 0;
      return "tk_insn_return";
    } else if (match(l, /^f?(b(g|l)(e|t))/) || match(l, /^f?b(ne|eq)/)) {
# jumps (indicate the end of prologue)
      L_TI["opc"] = L_tokens[1];
      L_TI["target"] = L_tokens[2,2];
      return "tk_insn_jmp";
    } else if (match(l, /^bis \$30,\$30,\$15/)) {
# bis $30,$30,$15
      L_TI["sign"] = "+";
      L_TI["offset"] = "0";
      return "tk_insn_setup_fp";
    } else if (match(l, /^bis \$15,\$15,\$30/)) {
# bis $15,$15,$30
      L_TI["sign"] = "+";
      L_TI["offset"] = "0";
      L_TI["base"] = "$15";
      L_TI["dest"] = "$30";
      return "tk_insn_reset_sp";
    } else if (match(l, /^ldgp \$29,0\(\$2[67]\)/)) {
      return "tk_insn_load_gp";
    } else if (NF == 0 || match(l, /^\./)) {
# blank line or directive
      return 0;			# uninteresting line
    } else {
# other instructions
      return "tk_insn_other";
    }
  }
}

function tokenize_label(name)
{
# look if this label appear after some spaces at the beginning of line
# if so, we unconditionally regard this label as a compiler label (a label
# that does not corresponding to a function name)
  if (match($0, /^[[:space:]]/)) {
    type = "compiler";
  } else {
    type = symbol_type(name);
  }
  L_TI["name"] = name;
  L_TI["type"] = type;
  if (type == "C" && 
      index(C_md_asm_header_labels, C_symbol_name(name) " ")) {
    return "tk_label_asm_header";
  } else if (index(C_md_asm_header_labels, name " ")) {
    return "tk_label_asm_header";
  } else if (type == "C") {
    return "tk_label_C";
  } else if (type == "compiler") {
    return "tk_label_compiler";
  } else {
    return 0;			# uninteresting line
  }
}

##
## break a line into array L_tokens
## e.g., add x,y,z ==> L_tokens[1] = "add", L_tokens[2] = "x,y,z"
##                     L_tokens[1,1] = "add"
##                     L_tokens[2,1] = "x", L_tokens[2,2] = "y", 
##                     L_tokens[2,3] = "z", 
## Sep 13, 99
## L_flat_tokens is a "flattened" representation of the same information
## e.g., add x,y,z ==> L_flat_tokens[1] = add
##                     L_flat_tokens[2] = "x"
##                     L_flat_tokens[3] = "y"
##                     L_flat_tokens[4] = "z"
## NOTE: L_tokens are maintained for historical reasons.
## eventually, all uses of L_tokens must go away and we only maintain 
## L_flat_tokens

function basic_tokenize_line(l,
			     a, i, j, ai, t, m, n)
{
  split(l, a);
  m = 0;
  for (i in a) {
    if (i > m) m = i;
    L_tokens[i] = a[i];
    split(a[i], ai, /,/);
    for (j in ai) {
      L_tokens[i,j] = ai[j];
    }
  }
  t = 1;
  for (i = 1; i <= m; i++) {
    split(a[i], ai, /,/);
    n = 0;
    for (j in ai) {
      if (j > n) n = j;
    }
    for (j = 1; j <= n; j++) {
      if (!match(L_tokens[i,j], /^[[:blank:]]*$/)) {
	if (C_dbg>1) {
	  dbg_print("L_tokens[" i "," j "] = " L_tokens[i,j]);
	}
	L_flat_tokens[t++] = L_tokens[i,j];
      }
    }
  }
  
  if (C_dbg>1) {
    for (i = 1; i < t; i++) {
      dbg_print("L_flat_tokens[" i "] = " L_flat_tokens[i]);
    }
  }
}

##
## tokenize line and return the token type ("tk_...")
## skip uninteresting lines
## 

function tokenize_line(\
		       r, x)
{
  delete L_tokens;
  delete L_flat_tokens;
  delete L_TI;
  x = match($0, /[^[:space:]].*/);
  if (x == 0) {
    l = "";
  } else if (x > 1) {
    l = substr($0, RSTART);
  } else {
    l = $0;
  }
  L_tokens[0] = l;
  L_flat_tokens[0] = l;
  basic_tokenize_line(l);
  if (C_cpu == "i386") {
    r = i386_tokenize_line(l);
  } else if (C_cpu == "sparc") {
    r = sparc_tokenize_line(l);
  } else if (C_cpu == "mips") {
    r = mips_tokenize_line(l);
  } else if (C_cpu == "alpha") {
    r = alpha_tokenize_line(l);
  } else wrong_port("cpu");
  if (C_dbg>0) dbg_print("{  lookahead == \"" l "\" (" r ")  }");
  token_check();
  return r;
}

function token_check()
{
  
}

##
## symbol manipulation
##

function symbol_type(s,
		     r)
{
  if (C_md_C_symbol_prefix != "" && 
      index(s, C_md_C_symbol_prefix) == 1) {
    r = "C";
  } else if (C_md_compiler_symbol_prefix != "" && 
	     index(s, C_md_compiler_symbol_prefix) == 1) {
    r = "compiler";
  } else if (match(s, /^[[:digit:]]/)) {
# mips compiler: "1:"
    r = "compiler";
  } else if (C_md_C_symbol_prefix == "") {
    r = "C";
  } else {
    r = "unknown";
  }
  if(C_dbg>1)dbg_print("symbol_type(" s ") == " r);
  return r;
}

function C_symbol_name(s)
{
  if (C_md_C_symbol_prefix != "") {
    return substr(s, length(C_md_C_symbol_prefix) + 1);
  } else return s;
}

function compiler_symbol_name(s)
{
  if (C_md_compiler_symbol_prefix != "") {
    return substr(s, length(C_md_compiler_symbol_prefix) + 1);
  } else return s;
}

function make_compiler_symbol(s)
{
  return C_md_compiler_symbol_prefix s;
}

function make_C_symbol(s)
{
  if (C_md_C_symbol_prefix != "") {
    return C_md_C_symbol_prefix s;
  } else return s;
}

##
## 0098ab ==> 9 * 16^3 + 8 * 16^2 + 10 * 16 + 11
##

function read_hex(s,
		  k, l, r, c)
{
  if (match(s, /[0-9a-f][0-9a-f]*$/)) {
    k = RSTART;
    l = RLENGTH;
    v = 0;
    r = 1;
    for (i = l - 1; i >= 0; i--) {
      c = substr(s, k + i, 1);
      if (!match(c, /[[:digit:]]/)) {
	if (c == "a") c = 10;
	else if (c == "b") c = 11;
	else if (c == "c") c = 12;
	else if (c == "d") c = 13;
	else if (c == "e") c = 14;
	else if (c == "f") c = 15;
      }
      v += c * r;
      r *= 16;
    }
    if(C_dbg>0) dbg_print("read_hex(" s ") ==> " v);
    return v;
  } else {
    fatal(s " is not a hexnum");
  }
}

function write_hex(v)
{
  return sprintf("%08x", v);
}

##
## we often want to put a name that is not a valid C symbol in a 
## generated C file. for example, we went to put a name ".Lxxx" in C.
## To do this, we convert any invalid character into Qnnn where nnn is 
## an ascii code for the invalid character, and we later revert to the
## original name when we look at the compiled assembly.
## Finally we prefix the converted string by "QQQ" or "QQ", the former
## indicating that this should be decoded and the later that this should not.
## the parameter TO_DECODE decides which prefix should be attached.
## e.g., if TO_DECODE == 1, the encoding and decoding process looks like:
##  .Lxxx ===> (convert) QQQQ046Lxxx in C file 
##        ===> (compile) Q046Lxxx in asm file 
##        ===> (revert)  .Lxxx in asm file
##
##


function encode_name(name, to_decode,
		     prefix, orig_name, i, j, k, ijk)
{
  prefix = (to_decode ? "QQQ_" : "QQ_");
  if (match(name, /^[_0-9A-PR-Za-z]*$/)) {
    if (C_dbg) dbg_print(name " ==> " prefix name);
    return prefix name;
  } else {
    orig_name = name;
    if (index(name, "Q")) {
      gsub(/Q/, "Q121", name);
      if (match(name, /^[_0-9A-Za-z]*$/)) {
	if (C_dbg) dbg_print(orig_name " ==> " prefix name);
	return prefix name;
      }
    }
    ijk = 0;
    for (i = 0; i < 2; i++) {
      for (j = 0; j < 8; j++) {
	for (k = 0; k < 8; k++) {
	  if (ijk in C_conv) {
	    r = "\\" C_conv[ijk];
	    if (match(name, "\\" C_conv[ijk])) {
	      s = "Q" i j k;
	      gsub(r, s, name);
	      if (match(name, /^[_0-9A-Za-z]*$/)) {
		if (C_dbg) dbg_print(orig_name " ==> " prefix name);
		return prefix name;
	      }
	    }
	  }
	  ijk++;
	}
      }
    }
    if (match(name, /^[_0-9A-Za-z]*$/)) {
      if (C_dbg) dbg_print(orig_name " ==> " prefix name);
      return prefix name;
    } else {
      fatal("encoding failed for " orig_name " ==> " name);
    }
  }
}


function decode_name(name, 
		     orig_name, result_name)
{
  orig_name = name;
  if (!match(name, /^QQQ_/)) {
    return 0;
  }
  while(match(name, /Q[0-1][0-7][0-7]/)) {
    pp = 0;
    pp += 8 * 8 * substr(name, RSTART + 1, 1);
    pp += 8 * substr(name, RSTART + 2, 1);
    pp += substr(name, RSTART + 3, 1);
    c = sprintf("%c", pp);
    name = substr(name, 1, RSTART - 1) c substr(name, RSTART + 4);
  }
  if (!match(name, /^QQQ_/)) {
    fatal(name " is not an encoded name");
  }
  result_name = substr(name, 5);
  if(C_dbg>0)dbg_print(orig_name " ==> " result_name);
  return result_name;
}

function unique_compiler_name(name)
{
  return C_md_compiler_symbol_prefix "x" substr(name, length(C_md_compiler_symbol_prefix));
}

function encode_file_name(name)
{
  return name;
}

function decode_file_name(name)
{
  return name;
}

##
## enter scanner
##

function enter_scanner(s,
		       i, n, spaces)
{
  n = G_stack_depth;
  G_stack[n] = s;
  G_stack_depth = n + 1;
  spaces = "";
  if (C_dbg>0) {
    for (i = 0; i < n; i++) spaces = spaces " ";
    dbg_print(spaces "begin scanning " s);
  }
}

function leave_scanner(s,
		       ss, i, n, spaces)
{
  n = G_stack_depth - 1;
  ss = G_stack[n];
  if (ss != s) fatal("expect to pop " s ", but popped " ss);
  G_stack_depth = n;
  spaces = "";
  if (C_dbg>0) {
    for (i = 0; i < n; i++) spaces = spaces " ";
    dbg_print(spaces "end scanning " s);
  }
}

##
## some trivial stuff
##

function is_number(x)
{
  return match(x, /^-?[[:digit:]]+$/);
}

function warn(s)
{
  printf("%s:%d: warning: %s\n", FILENAME, FNR, s) > C_errout;
}

function wrong(s)
{
  printf("%s:%d: error: %s\n", FILENAME, FNR, s) > C_errout;
  exit(1);
}

function fatal(s)
{
  printf("%s:%d: FATAL ERROR: %s\n", FILENAME, FNR, s) > C_errout;
  exit(1);
}

function syntax_error(expected_tokens)
{
  printf("%s:%d: syntax error at \"%s\" (token type = %s).\n  Expected %s\n", 
	 FILENAME, FNR, L_flat_tokens[0], L_tok, expected_tokens) > C_errout;
  printf("  Parser stack:\n") > C_errout;
  for (i = G_stack_depth - 1; i >= 0; i--) {
    printf("  %d : %s\n", i, G_stack[i]) > C_errout;
  }
  exit(1);
}

function dbg_print(s)
{
  printf("%s DBG[%4d]: %s\n", C_md_comment_string, FNR, s);
}

function wrong_port(category)
{
  fatal("wrong " category " (config = " C_st_config ")");
}

function wrong_task()
{
  fatal("wrong task " C_task);
}

function cat2(a, b) { return a b; }
function cat3(a, b, c) { return a b c; }
function cat4(a, b, c, d) { return a b c d; }
function cat5(a, b, c, d, e) { return a b c d e; }

##
## real entry point
##

{
  if (NR == 1) {
    main();
    exit 0;
  } else {
    fatal("dont call next command\n");
  }
}
