;;;
;;;
;;;

(defvar *rd-current-race-record-line* 0)
(defvar *rd-race-records* "")

;;;
;;; a race record is represented as
;;; '("qs.c:123" "qs.c:220" "foo.c:10" "bar.c:20")
;;;

(defun rd-make-race-record (pt pa tt ta)
  (list pt pa tt ta))

;;;
;;; parse region
;;;

(defun rd-parse-region (b e)
  "(RD-PARSE-REGION B E): parses every lines between points B and E.
returns a vector of parsed lines. Each parse line is a race record."
  (interactive "r")
  (goto-char b)
  (let ((c 0)
	(r '()))
    ;; accumulate parsed lines
    (while (< (point) e)
      (setq r (cons (rd-parse-line) r))
      (forward-line 1)
      (setq c (+ c 1)))
    (message (format "%d lines parsed" c))
    (let ((v (make-vector c '())))
      ;; make them a vector
      (while (> c 0)
	(setq c (- c 1))
	(aset v c (car r))
	(setq r (cdr r)))
      v)))

;;;
;;; parse a line that begins at the point
;;;

(defun rd-parse-line ()
  ;; skip the header mtcamp_race_detected: 
  (re-search-forward "mtcamp_race_detected: " nil nil)
  (let* ((prev-thread (rd-next-source-info))
	 (prev-access (rd-next-source-info))
	 (this-thread (rd-next-source-info))
	 (this-access (rd-next-source-info)))
    (rd-make-race-record prev-thread prev-access this-thread this-access)))

;;;
;;; read "filename:line" from the point
;;;

(defun rd-next-source-info ()
  (re-search-forward "\\([^:]+:[^ ]+\\) " nil nil)
  (match-string 1))

;;;
;;;
;;;

(defvar *rd-current-race-record-arrows* '())

(defun rd-display-arrow (p arrow)
  (let ((o (make-overlay p (+ p (length arrow)))))
    (overlay-put o 'invisible t)
    (overlay-put o 'before-string arrow)
    (setq *rd-current-race-record-arrows* 
	  (cons o *rd-current-race-record-arrows*))))

(defun rd-delete-arrows ()
  (while *rd-current-race-record-arrows*
    (delete-overlay (car *rd-current-race-record-arrows*))
    (setq *rd-current-race-record-arrows*
	  (cdr *rd-current-race-record-arrows*))))

;;;
;;; si is a string that looks like "FILE:LINE"
;;; open FILE, goto LINE, and show ARROW at the beginning of the line
;;;

(defun rd-open-source-info (si arrow)
  (interactive "sFILENAME:LINE: ")
  (string-match "\\([^:]+\\):\\(.*\\)" si)
  (let ((file (match-string 1 si))
	(line (string-to-number (match-string 2 si))))
    (find-file-other-window file)
    (goto-line line)
    ;; show arrow 
    (rd-display-arrow (point) arrow)))

;;;
;;; show source code that corresponding to yth column of xth record.
;;; e.g., if the whole record looks like
;;; '(("a.c:10" "b.c:20" "c.c:30" "d.c:40")
;;;   ("aa.c:10" "bb.c:20" "cc.c:30" "dd.c:40")
;;;   ("aaa.c:10" "bbb.c:20" "ccc.c:30" "ddd.c:40")),
;;; (rd-show-this-source 0 2) shows 30th line of c.c
;;; (rd-show-this-source 1 3) shows 40th line of dd.c
;;;

(defun rd-show-this-source (x y arrow)
  (interactive)
  (rd-open-source-info (elt (elt *rd-race-records* x) y) arrow))

(defun rd-show-current-record ()
  (interactive)
  (if (< *rd-current-race-record-line* (length *rd-race-records*))
      (progn
	(rd-delete-arrows)
	(rd-show-this-source *rd-current-race-record-line* 0 "=:")
	(rd-show-this-source *rd-current-race-record-line* 2 "-:")
	(rd-show-this-source *rd-current-race-record-line* 1 "=>")
	(rd-show-this-source *rd-current-race-record-line* 3 "->")
	t)
      (progn
	(message "End of records")
	(beep)
	nil)))
  
(defun rd-show-current-record-and-next ()
  (interactive)
  (if (rd-show-current-record)
      (setq *rd-current-race-record-line* 
	    (+ *rd-current-race-record-line* 1))))

(defun rd-show-current-record-and-prev ()
  (interactive)
  (if (rd-show-current-record)
      (setq *rd-current-race-record-line* 
	    (- *rd-current-race-record-line* 1))))

;;;
;;; open race record file
;;; format:
;;; mtcamp_race_detected: a.c:5 b.c:3 c.c:4 d.c:2 
;;; mtcamp_race_detected: q.c:7 b.c:8 z.c:9 d.c:10 
;;; mtcamp_race_detected: q.c:10 y.c:15 t.c:9 f.c:2 
;;; mtcamp_race_detected: r.c:1 x.c:2 s.c:5 d.c:8 
;;;
;;; each line must have a space at the end
;;;

(defun rd-visit-race-records (file)
  (interactive "sfile to visit : ")
  (let ((b (find-file file)))
    (setq *rd-race-records* (rd-parse-region (point-min) (point-max)))
    (setq *rd-current-race-record-line* 0)
    (kill-buffer b))
  (rd-show-current-record-and-next))

(define-key ctl-x-map "\C-a" 'rd-show-current-record-and-next)
