
## 
## Copyright (c) 1999 by Kenjiro Taura, Akinori Yonezawa. All rights reserved.
## Copyright (c) 1999 by Yoshihiro Oyama, Toshio Endo. All rights reserved.
## Copyright (c) 1999 by Kunio Tabata. All rights reserved.
## Copyright (c) 1999 by Mitsubishi Research Institute.  All rights reserved.
## Copyright (c) 1999 by Information-technology Promotion Agency.  All rights reserved.
##
## THIS MATERIAL IS PROVIDED AS IS, WITH ABSOLUTELY NO WARRANTY EXPRESSED
## OR IMPLIED.  ANY USE IS AT YOUR OWN RISK.
##
## Permission is hereby granted to use or copy this program
## for any purpose,  provided the above notices are retained on all copies.
## Permission to modify the code and to distribute modified code is granted,
## provided the above notices are retained, and a notice that the code was
## modified is included with the above copyright notice.
##

#
# stgcc.awk 
#
# usually used from a shell script stgcc, which does:
#
# echo $* | gawk -f stgcc.awk 
#
# stgcc is intended to be a complete frontend for gcc, which accept
# all the arguments accepted by gcc. the behaviour is also the same as
# gcc, except that assembly files are postprocessed to support 
# StackThreads primitives. currently, either gcc or g++ are assumed.
# 
# while many options are simply passed through gcc, we still must do 
# a substantial amount of reasoning about given arguments. this script
# basically works as follows:
#
# (1) parse arguments:
# we record types of input files (C, C++, header, objects, etc.) and
# where to stop the whole process (i.e., after generating
# assembly, after generating objects, etc.)
#
# (2) compilation:
# if any file needs to be compiled into an object file, we invoke the
# underlying C compiler (gcc or g++) to generate assembly files, with 
# mostly unmodified command line options (we still have to ged rid of 
# linker input files (.o or -lxx)). as a result, some assembly files are
# generated. we keep track of which assembly file is generated from which
# input file.
#
# for each assembly file generated in this way, we postprocess it.
#
# for each assembly file generated by postprocessing and each assembly file 
# specified in the command line, we assemble it.
#
# for each object files generated in the above step and object/library files 
# specified in the command line, we link them together.

function machine_config(c, o)
{
  CPU = c;
  OS = o;

  supported_config["i386","winnt"] = 1;
  supported_config["i386","solaris"] = 1;
  supported_config["i386","linux"] = 1;
  supported_config["mips","irix"] = 1;
  supported_config["sparc","solaris"] = 1;
  supported_config["alpha","osf1"] = 1;
  supported_config["alpha","linux"] = 1;

  if (CPU == 0) st_cpu_unset_error();
  if (OS == 0) st_os_unset_error();
  if (!(CPU SUBSEP OS in supported_config)) unknown_configuration();

}

function machine_specific_C_options(\
				    opts)
{
  if (CPU == "i386") {
# return " -fno-defer-pop";
    return "";
  } else if (CPU == "mips") {
    return " -fno-omit-frame-pointer -mips2";
  } else if (CPU == "sparc") {
    opts = " -mflat -Dsetjmp=asm_mflat_setjmp -Dsigsetjmp=asm_mflat_sigsetjmp -D_setjmp=asm_mflat__setjmp -Wa,-xarch=v8plus";
    if (st_safety_options_are_ok) opts = opts " -mcallee-copies-struct-args";
    return opts;
  } else if (CPU == "alpha") {
    return " -fno-omit-frame-pointer";
  } else unknown_cpu_error();
}

function machine_specific_asm_options()
{
  if (CPU == "i386" || CPU == "alpha") {
    return "";
  } else if (CPU == "mips") {
    return " -mips2";
  } else if (CPU == "sparc") {
    return "-Wa,-xarch=v8plus";
  } else unknown_cpu_error();
}

function machine_specific_fixed_reg_options()
{
  if (CPU == "i386") {
    fixed_reg = "%ebx";
  } else if (CPU == "mips") {
    fixed_reg = "\\$23";
  } else if (CPU == "sparc") {
    fixed_reg = "%l7";
  } else if (CPU == "alpha") {
    fixed_reg = "\\$14";
  } else unknown_cpu_error();
  return "-ffixed-" fixed_reg;
}

function machine_specific_no_cs_options()
{
  result = "";
  if (CPU == "i386") {
    cs_regs_str = "%esi %edi";
  } else if (CPU == "mips") {
    ics_regs_str = "\\$16 \\$17 \\$18 \\$19 \\$20 \\$21 \\$22";
    fcs_regs_str = "\\$f20 \\$f22 \\$f24 \\$f26 \\$f28 \\$f30";
    cs_regs_str = ics_regs_str " " fcs_regs_str;
  } else if (CPU == "sparc") {
    cs_regs_str = "%l0 %l1 %l2 %l3 %l4 %l5 %l6 %i0 %i1 %i2 %i3 %i4 %i5";
  } else if (CPU == "alpha") {
    ics_regs_str = "\\$9 \\$10 \\$11 \\$12 \\$13";
    fcs_regs_str = "\\$f2 \\$f3 \\$f4 \\$f5 \\$f6 \\$f7 \\$f8 \\$f9";
    cs_regs_str = ics_regs_str " " fcs_regs_str;
  } else {
    unknown_cpu_error();
  }
  split(cs_regs_str, cs_regs, " ");
  n = 0;
  for (x in cs_regs) n++;
  for (i = 1; i <= n; i++) {
    result = result " -ffixed-" cs_regs[i];
  }
  return result;
}

# prohibit certain kind of optimizations
function no_opt_options(opts)
{
  opts = "";
  if (inline_functions_is_ok == 0) {
    opts = opts " -fno-inline-functions";
  }
  if (default_inline_is_ok == 0) {
    opts = opts " -fno-default-inline";
  }
  if (st_safety_options_are_ok) {
    opts = opts " -fcalls-clobber-sp";
  }
  return opts;
}

function cpp_common_options()
{
  opts = sprintf("-I%s/include -DSTHREADS", ST_DIR);
  if (no_postprocess) {
    opts = opts " -DSTHREADS_NOPP"
  } 
  if (alloca_is_ok == 0) {
    opts = opts " -Dalloca=st_alloca";
  }
  return opts;
}

function os_specific_cpp_options()
{
  if (OS == "solaris" || OS == "linux") {
    return "-D_REENTRANT";
  } else if (OS == "irix") {
    return "-D_SGI_MP_SOURCE";
  } else if (OS == "osf1") {
    return "-D_THREAD_SAFE";
  } else {
    return "";
  }
}

function os_specific_stlink_options(\
				    x)
{
  if (ST_THREAD_PKG == "st_solaris_thread") {
    x = "-lthread";
  } else if (ST_THREAD_PKG == "st_pthread") {
    x = "-lpthread";
  } else if (ST_THREAD_PKG == "st_old_pthread") {
    x = "-lpthreads";
  } else if (ST_THREAD_PKG == "st_sfork_thread" ||
	     ST_THREAD_PKG == "st_nt_thread" ||
	     ST_THREAD_PKG == "st_no_thread") {
    x = "";
  } else {
    unknown_thread_pkg_error();
  }

  if (OS == "solaris") {
    x = x " -lposix4";
  } else if (OS == "irix" || OS == "linux" || OS == "osf1" || OS == "winnt") {
    
  } else unknown_os_error();

  return x;
}

function set_compiler_options()
{
  cpp_common = cpp_common_options();

  edg_options = cpp_common sprintf(" --preinclude %s -DMTCAMP", ST_DIR "/include/mtcamp.h");
  cpp_options = cpp_common " " os_specific_cpp_options();
  C_common_options = cpp_options " " no_opt_options();
  C_options = C_common_options " " machine_specific_fixed_reg_options() " " machine_specific_C_options();
  C_no_cs_options = C_options " " machine_specific_no_cs_options();
  asm_common_options = "";
  asm_options = asm_common_options " " machine_specific_asm_options();
  asm_cpp_options = asm_options " " cpp_options;

  stlink_common_options = "-L" ST_DIR "/lib";
  stlink_os_options = os_specific_stlink_options();
  if (libst_a == 0) libst_a = "st";
  if (libstext_a == 0) libstext_a = "stext";
}  

# frontend of `system' command.
function my_system(cmd)
{
  if (dont_execute || show_commands) {
    printf("%s\n", cmd) > "/dev/stderr";
  }
  if (!dont_execute) {
    x = system(cmd);
# I don't know what's wrong here.
# On windows NT, we must do this to flush buffered output. 
# (otherwise, output from CMD is not printed until we issue next command)
    if (OS == "winnt") {
      system("echo -n");		# flush buffered output
    }
    if (show_commands) {
      if (x == 0) {
	printf("OK\n") > "/dev/stderr";
      } else {
	printf("NOT OK\n") > "/dev/stderr";
      }
    }
    return x;
  }
}

# 1 if FILE exists
function file_exists(file)
{
  cmd = sprintf("ls %s 2> /dev/null > /dev/null", file);
  if (my_system(cmd) == 0) return 1;
  else return 0;
}

# copy x to y. return status of the copy command (0 if success)
function copy_file(x, y)
{
  cmd = sprintf("cp %s %s", x, y);
  if (my_system(cmd) == 0) return 0;
  else return 1;
}

# return 1 if ARG should not be given to compilation phase (without linking)
# currently it is simply -lxx things.
function is_linker_option(arg)
{
  if (match(arg, /^-l.*/)) return 1;
  else if (arg == "--leave_link_file") return 1;
  else return 0;
}

function is_output_option(arg)
{
  return arg == "-o";
}

function is_language_select_option(arg)
{
  return arg == "-x";
}

function is_where_to_stop_option(arg)
{
  if (match(arg, /^-[ScE]/)) return 1;
  return 0;
}

function is_edg_only_option(arg) 
{
  if (match(arg, /^---.*/)) return 1;
  return 0;
}

function translate_edg_only_option(arg)
{
# strip first -
  return substr(arg, 2);
}

# options common to EDG and CPP
function is_edg_cpp_common_option(arg)
{
  if (match(arg, /^-[CPME]/)) return 1;
  if (match(arg, /^-[DUI].*/)) return 1;
  return 0;
}

# return 1 if this option takes an argument (i.e., the next argument should 
# not be considered as an input file)
function is_take_arg_option(arg)
{
  if (arg == "-o" || arg == "-x") {
    return 1;
  } else {
    return 0;
  }
}

# return 1 if ARG is an option specific to stgcc and thus should not be
# passed to the underlying compiler
function is_stgcc_option(arg)
{
  if (match(arg, /^--(leave_tmps|show_commands|dont_execute|dbg)$/)) return 1;
  if (match(arg, /^--(no_callee_saves|no_postprocess)$/)) return 1;
  if (match(arg, /^--(default_inline_is_ok|inline_functions_is_ok|alloca_is_ok)$/)) return 1;
  if (match(arg, /^--(x?mtcamp|xsgc|xmem_trace)$/)) return 1;
  if (match(arg, /^--addlib=.*/)) return 1;
  if (match(arg, /^--sal.*/)) return 1;
  if (match(arg, /^--compiler=.*/)) return 1;
  return 0;
}

# check 
function is_dangerous_option(arg)
{
  if (match(arg, /^-f(call-saved-|call-used-|fixed-|omit-frame-pointer)/)) {
    return 1;
  } else if (match(arg, /^-finline-functions/)) {
    if (inline_functions_is_ok) return 0;
    else return 1;
  } else if (match(arg, /^-fdefault-inline/)) {
    if (default_inline_is_ok) return 0;
    else return 1;
  } else return 0;
}

# return its language type from file name
# (.cc --> C++, .c --> C etc.)
function default_language(file)
{
  if(match(file, /\.[^\.]*$/)) {
    ext = substr(file,RSTART+1);
    lang = default_language_from_ext[ext];
    if (lang) {
      return lang;
    } else {
      return "linker_input";
    }
  } else {
    return "linker_input";
  }
}

# xxx.yyy --> xxx.ext
# or xxx --> xxx.ext
function make_output_file_name(c_path, ext)
{
# remove directory portion
  match(c_path, /[^/]*$/);
  base = substr(c_path, RSTART);
  if(match(base, /\.[^\.]*$/)) {
    return substr(base,1,RSTART) ext;
  } else {
    return (base "." ext);
  }
}

function default_edg_output(c_path, lang)
{
  if (lang == "c" || lang == "cpp-output") {
    ext = "int.c";
  } else if (lang == "c++" || lang == "c++-cpp-output") {
    ext = "int.cc";
  }
  return make_output_file_name(c_path, ext);
}

function default_asm_output(c_path)
{
  return make_output_file_name(c_path, "s");
}

function default_nopp_asm_output(c_path)
{
  return make_output_file_name(c_path, "t");
}

function default_obj_output(c_path)
{
  return make_output_file_name(c_path, "o");
}

# when arguments are not right (e.g., when multiple -o are given), 
# we still invoke the underlying compiler to get the right error message
function invoke_error_command()
{
  cmd = sprintf("%s %s", compiler, all_args);
  err = my_system(cmd);
  if (!err) {
    printf("warning: %s did not end up with an error\n", cmd) > "/dev/stderr";
  }
  return err;
}

# init constants (tables)
function init_constants()
{
  delete default_language_from_ext;
  default_language_from_ext["c"] = "c";
  default_language_from_ext["i"] = "cpp-output";
  default_language_from_ext["ii"] = "c++-cpp-output";
  default_language_from_ext["m"] = "objective-c";
  default_language_from_ext["h"] = "c-header";
  default_language_from_ext["cc"] = "c++";
  default_language_from_ext["cxx"] = "c++";
  default_language_from_ext["cpp"] = "c++";
  default_language_from_ext["C"] = "c++";
  default_language_from_ext["s"] = "assembler";
  default_language_from_ext["S"] = "assembler-with-cpp";

  stop_after_cpp = 1;
  stop_after_compile = 2;
  stop_after_assemble = 3;
  stop_after_link = 4;

  delete supported_langs;
  supported_langs["c"] = 1;
  supported_langs["objective-c"] = 1;
  supported_langs["c++"] = 1;
  supported_langs["c-header"] = 1;
  supported_langs["cpp-output"] = 1;
  supported_langs["c++-cpp-output"] = 1;
  supported_langs["assembler"] = 1;
  supported_langs["assembler-with-cpp"] = 1;
  supported_langs["linker_input"] = 1;

  delete compiled_langs;
  compiled_langs["c"] = 1;
  compiled_langs["objective-c"] = 1;
  compiled_langs["c++"] = 1;
  compiled_langs["cpp-output"] = 1;
  compiled_langs["c++-cpp-output"] = 1;

  delete edged_langs;
  edged_langs["c"] = 1;
  edged_langs["c++"] = 1;
  edged_langs["cpp-output"] = 1;
  edged_langs["c++-cpp-output"] = 1;

  delete assemble_only_langs;
  assemble_only_langs["assembler"] = 1;
  assemble_only_langs["assembler-with-cpp"] = 1;

}

# init variables
function init_variables()
{
# 1 if we want to leave temporary files for postprocessing (set when 
# --leave_tmps is given)
  leave_tmps = 0;
# 1 if we don't want to execute command (useful for debugging)
  dont_execute = 0;
# 1 if we show command lines
  show_commands = 0;
# 1 if the compiler should not use any callee-save register
  no_callee_saves = 0;
# 1 if the compiler should not postprocess assembly files
  no_postprocess = 0;
# 1 if EDG is used
  do_edg_preprocess = 0;
# options that must come AFTER -lstext -lst (normally they are additional
# libraries used BY libst.a
  addlib_options = "";
# set when you are sure that -fno-default-inlines and -finline-functions are
# unnecessary (wizard only)
  inline_functions_is_ok = 0;
  default_inline_is_ok = 0;
  st_safety_options_are_ok = ST_GCC_IS_PATCHED;
# set when you are sure that system-provided alloca is OK (wizard only)
  alloca_is_ok = 0;

# by default, the process won't be stopped until we have done linking
  where_to_stop = stop_after_link;
# set to 1 when we find any file that needs compilation into .o file
  need_compilation = 0;

# all arguments (except for stgcc-specific ones) are accumulated
  all_args = "";

# args needed to preprocess input files by EDG
  edg_args = "";
# args needed to compile input files into assembly
  compile_args = "";
# args needed to generate object files from assembly
  assemble_args = "";
# args needed to link all files
  delete link_args;
  n_link_args = 0;

# files[l,i] is i th filename of language L
  delete xfiles;
  delete n_xfiles;
  for (l in supported_langs) n_xfiles[l] = 0;
  n_total_input_files = 0;

# non_postprocessed_assembly_files[C_file_name] is the name of the 
# (non postprocessed) assembly file generated from input file C_file_name
  delete non_postprocessed_assembly_files;
# postprocessed_assembly_files[C_file_name] is the name of the 
# (postprocessed) assembly file generated from input file C_file_name
  delete postprocessed_assembly_files;
# object_files[C_file_name/asm_file_name] is the name of the 
# object file generated from C_file_name/asm_file_name.
  delete object_files;

# output file specified by "-o"
  output_file = 0;

# language currently assumed
  language = "none";

# 1 if the next argument should be an argument to the previous flag thus
# must not be interpreted as an input file
  require_arg = 0;

  tmpfile_count = 0;
}

function min(x, y)
{
  if (x < y) return x;
  else return y;
}

# parse all arguments
# 0 : OK
# 1 : a command line error is detected, but we still should invoke 
#     gcc, so that it should display familiar diagnostic messages
# 2 : a command line error not understandable gcc is detected

function parse_args(i) # i : local variable
{
# set when we know there is an error in the command line
  command_line_error = 0;
  minus_appeared = 0;
  for (i = 0; i < ARGC; i++) {
# accumulate all args in ARGS
#    if (i == NF && OS == "winnt") {
# gnuwin b19 does not chop Ctrl-M
#      arg = substr($i,1,length($i)-1);
#    } else {
#      arg = $i;
#    }
    arg = ARGV[i];
# skip arguments that come before "-"
    if (!minus_appeared) {
      if (arg == "-") {
	minus_appeared = 1;
      }
      continue;
    }

# if we find a dangerous option that violates our assumption, immediately
# return 
    if (is_dangerous_option(arg)) {
      printf("option %s is prohibited by stgcc (abort)\n", arg) > "/dev/stderr";
      return 2;
    }
    if (!is_stgcc_option(arg)) all_args = all_args " " arg;
    if (command_line_error) {
# when command_line_error is detected we still continue parsing
# to pass all the given arguments to gcc.
      continue;
    }

# check if this is an option
    if (match(arg, /^-.*/)) {
# check options that affect where to stop compilation and options specific to
# stgcc
      if (arg == "-c") {
	where_to_stop = min(where_to_stop, stop_after_assemble);
      } else if (arg == "-S") {
	where_to_stop = min(where_to_stop, stop_after_compile);
      } else if (arg == "-E") {
	where_to_stop = min(where_to_stop, stop_after_cpp);
      } else if (arg == "--leave_tmps") {
	leave_tmps = 1;
      } else if (arg == "--show_commands") {
	show_commands = 1;
      } else if (arg == "--dont_execute") {
	dont_execute = 1;
      } else if (arg == "--no_callee_saves") {
	no_callee_saves = 1;
      } else if (arg == "--no_postprocess") {
	no_postprocess = 1;
      } else if (arg == "--inline_functions_is_ok") {
	inline_functions_is_ok = 1;
      } else if (arg == "--default_inline_is_ok") {
	default_inline_is_ok = 1;
      } else if (arg == "--alloca_is_ok") {
	alloca_is_ok = 1;
      } else if (arg == "--dbg") {
	leave_tmps = 1;
	show_commands = 1;
      } else if (arg == "--mtcamp" || arg == "--xmtcamp") {
# --mtcamp is for backward compatibility. it should eventually be removed
	do_edg_preprocess = 1;
      } else if (arg == "--xsgc") {
# synonym for --addlib=$SGC_DIR/sgc.a --libst_a=st_sgc --libstext_a=st_sgc
	if (SGC_DIR) {
	  libst_a = "st_sgc";
	  libstext_a = "stext_sgc";
	  addlib_options = addlib_options " " SGC_DIR "/sgc.a";
	  compile_args = compile_args " -I" SGC_DIR;
	  edg_args = edg_args " -I" SGC_DIR;
	} else {
	  printf("Environment variable SGC_DIR not defined (link will fail)\n") > "/dev/stderr";
# we just go ahead
	}
      } else if (arg == "--xmem_trace") {
# synonym for --mtcamp ---mem_trace 
	do_edg_preprocess = 1;
	edg_args = edg_args " --mem_trace";
	if (MTC_DIR) {
	  addlib_options = addlib_options " " MTC_DIR "/fe/rd/lib/librace.a";
	} else {
	  printf("Environment variable MTC_DIR not defined (link will fail)\n") > "/dev/stderr";
	}
      } else if (match(arg, /^--addlib=.*/)) {
# used in a form like this: "--addlib=sgc.a"
	addlib_options = addlib_options " " substr(arg, length("--addlib=")+1);
      } else if (match(arg, /^-sal.*/)) {
# used in a form like this: "-salX" (like -lX)
	addlib_options = addlib_options " -l" substr(arg, length("-sal")+1);
      } else if (match(arg, /^--libst_a=.*/)) {
	libst_a = substr(arg, length("--libst_a=")+1);
      } else if (match(arg, /^--libstext_a=.*/)) {
	libstext_a = substr(arg, length("--libstext_a=")+1);
      } else if (match(arg, /^--compiler=.*/)) {
	compiler = substr(arg, length("--compiler=")+1);
      }

# this is an option (argument that begins with -)

# we get rid of -o output_file, stgcc-specific options, -E, -S, -c things,
# and -x ... option.
# types of command lines
      if (!is_stgcc_option(arg) && !is_output_option(arg) &&
	  !is_edg_only_option(arg) &&
	  !is_where_to_stop_option(arg) && !is_language_select_option(arg)) {
	link_args[n_link_args] = arg;
	n_link_args++;

# we further remove -l... things from the command line that generates
# assembly from C/C++ files	
	if (!is_linker_option(arg)) {
	  compile_args = compile_args " '" arg "'";
	  assemble_args = assemble_args " '" arg "'";
	}
      }

      if (is_edg_only_option(arg)) {
# EDG-only option begins with ---. remove the first -.
	edg_args = edg_args " " translate_edg_only_option(arg);
      } else if (is_edg_cpp_common_option(arg)) {
	edg_args = edg_args " '" arg "'";
      }

# when we find things like -o tell the next iteration that the next 
# argument should not be interpreted as input file name
      if (is_take_arg_option(arg)) {
	require_arg = 1;
	require_arg_flag = arg;
      } else {
	require_arg = 0;
      }

    } else if(require_arg) {
# this argument does not begin with `-', but it is still not an input file,
# because previous iteration is a flag that takes an argument (like -x 
# ... or -o ...)

# do appropriate actions according to the previous arg (-x, -o ...)
      if (is_language_select_option(require_arg_flag)) {
	language = arg;
      } else if (is_output_option(require_arg_flag)) {
# it is error to speicify multipe -o s
	if (output_file) {
	  command_line_error = 1;
        } else {
	  output_file = arg;
	}
      }

# we get rid of `-o filename' from all types of command lines
      if (!is_output_option(require_arg_flag) &&
	  !is_language_select_option(require_arg_flag)) {
# currently we never reach here. in future, we may add an option that
# takes an argument and needs to be added to the command line
	compile_args = compile_args " '" arg "'";
	assemble_args = assemble_args " '" arg "'";
	link_args[n_link_args] = arg;
	n_link_args++;
      }

      require_arg_flag = 0;
      require_arg = 0;
    } else {
# this is an input file. record it into xfiles array and setup arguments.
      if (language == "none") {
	l = default_language(arg);
      } else {
	l = language;
      }
      if (l in supported_langs) {
# if arg is not a linker input (e.g., .c file), arg is later translated
# into generated object name. for now, we simply store it into link_args.
	link_args[n_link_args] = arg;
	n_link_args++;
	if (l in compiled_langs) {
	  need_compilation = 1;
	}
	xfiles[l,n_xfiles[l]] = arg;
	n_xfiles[l]++;
	n_total_input_files++;
      } else {
	command_line_error = 1;
      }
    }
  }
# it is an error to speicify -o and multiple input files
  if (n_total_input_files > 1 && 
      where_to_stop != stop_after_link && output_file) {
    command_line_error = 1;
  }
  return command_line_error;
}

# preprocess SRC_FILE by EDG and generate DEST_FILE
function edg_file(lang, in_file, dest_file,
		  cmd, err)
{
# signal an error here if MTCAMP is not defined
  if (MTC_DIR == 0) {
    printf("Environment variable MTC_DIR not defined\n") > "/dev/stderr";
    return 1;
  }

# here is a tentative way in which we handle GCC-specific extensions that 
# must be generated by EDG. we first process st.h and gather lines 
# following "pragma gccext". these lines are summarized in a temporary file
# below. next we run EDG. finally output generated by EDG is patched
# using the summary file. eventually this pre/post-processing should go away.

# preprocess st.h
  pragma_gcc_ext = gen_temp_file("gcc_ext", "xx");
  cmd = sprintf("gawk -f %s/bin/gccext.awk TASK=preprocess PID=%s %s/include/st.h > %s",
		ST_DIR, PID, ST_DIR, pragma_gcc_ext);
  err = my_system(cmd);
  if (err) {
    my_system("rm -f " pragma_gcc_ext);
    return err;
  }

# run EDG.
  if (lang == "c" || lang == "cpp-output") {
    lang_opt = "--c";
  } else if (lang == "c++" || lang == "c++-cpp-output") {
    lang_opt = "--c++";
  } else {
    printf("invalid lang %s\n", lang) > "/dev/stderr";
    return 1;
  }

  tmp_dest_file = dest_file ".x";
# tmp_dest_file must always be used (EDG does not like --gen_c_file xxx.int.c)
  gen_c = "--gen_c_file_name " tmp_dest_file;
  
  cmd = sprintf("%s %s %s %s %s %s", 
		edg, lang_opt, edg_options, edg_args, 
		gen_c, in_file);
  err = my_system(cmd);
  if (err) {
    my_system("rm -f " pragma_gcc_ext);
    return err;
  }

# postprocess generated output
  if (file_exists(tmp_dest_file)) {
    cmd = sprintf("gawk -f %s/bin/gccext.awk TASK=postprocess REPLACED_LINES=%s %s > %s", 
		  ST_DIR, pragma_gcc_ext, tmp_dest_file, dest_file);
    err = my_system(cmd);
    my_system("rm -f " pragma_gcc_ext " " tmp_dest_file);
  }
  return err;
}

function preprocess_file(lang, in_file,
			 tmp)
{
  if (do_edg_preprocess) {
# when --mtcamp is also given, output is EDG output.
# run EDG and cat output
    tmp = gen_temp_file("cpp", "i");
    err = edg_file(lang, in_file, tmp);
    if (file_exists(tmp)) {
      my_system("cat " tmp);
      my_system("rm -f " tmp);
    }
    return err;
  } else {
# otherwise simply invoke the underlying C compiler
    cmd = sprintf("%s %s -E %s %s",
		  compiler, cpp_options, compile_args, in_file);
    return my_system(cmd);
  }
}

function preprocess_files(\
			  i, n)
{
  if(show_commands)print "preprocess_files";
  for (l in compiled_langs) {
    n = n_xfiles[l];
    for (i = 0; i < n; i++) {
      err = preprocess_file(l, xfiles[l,i]);
      if (err) return 1;
    }
  }
  return 0;
}

# compile IN_FILE and generate NOPP_ASM_FILE
# LANG is a language name listed in compiled_langs

function compile_file(lang, in_file, nopp_asm_file,
		      edg_pp_file, src_file, err)
{
  if(show_commands)print "compile_file (" lang ") " in_file " => " nopp_asm_file;
# record the fact that NOPP_ASM_FILE came from IN_FILE
  non_postprocessed_assembly_files[in_file] = nopp_asm_file;

# EDG preprocess
# edg_pp_file : file generated by EDG
# src_file : file feed to gcc

  if (do_edg_preprocess) {
    if (leave_tmps) {
      edg_pp_file = default_edg_output(in_file, lang);
    } else {
      if (lang == "c" || lang == "cpp-output") {
	ext = "c";
      } else if (lang == "c++" || lang == "c++-cpp-output") {
	ext = "cc";
      }
      edg_pp_file = gen_temp_file("edg", "int." ext);
    }
    src_file = edg_pp_file;
    
    err = edg_file(lang, in_file, edg_pp_file);
    if (err) {
      if (leave_tmps == 0) {
	cmd = sprintf("rm -f %s", edg_pp_file);
	my_system(cmd);
      }
      return err;
    }
    
    if (lang == "c" || lang == "cpp-output") {
      out_lang = "c";
    } else if (lang == "c++" || lang == "c++-output") {
      out_lang = "c++";
    }
  } else {			# do_edg_preprocess == 0
    edg_pp_file = 0;
    src_file = in_file;
    out_lang = lang;
  }

  cmd = sprintf("%s %s %s -S -x %s %s -o %s", 
		compiler, 
		(no_callee_saves ? C_no_cs_options : C_options), 
		compile_args, out_lang, src_file, nopp_asm_file);
  err = my_system(cmd);

  if (edg_pp_file && (leave_tmps == 0)) {
    cmd = sprintf("rm -f %s", edg_pp_file);
    my_system(cmd);
  }
  return err;
}

function gen_temp_file(kw, ext)
{
  return "TMP_st_" kw "_" PID "_" tmpfile_count++ "." ext;
}

# compile every file of type LANG, no matter LANG is explicitly is given
# by -x or is implicitly determined
# LANG is either a language name listed in compiled_langs.
function compile_lang(lang,
		      i, n)
{
# compile every file whose language is explicitly specified with -x
  n = n_xfiles[lang];
  for (i = 0; i < n; i++) {
    in_file = xfiles[lang,i];
    if (leave_tmps) {
      nopp_asm_file = default_nopp_asm_output(in_file);
    } else {
      nopp_asm_file = gen_temp_file("nopp", "t");
    }
    err_comp = compile_file(lang, in_file, nopp_asm_file);
    if (err_comp) return 1;
  }
  return 0;
}

# compile every file that needs compilation
function compile_files()
{
  if(show_commands)print "compile_files";
  for (l in compiled_langs) {
    err = compile_lang(l);
    if (err) return 1;
  }
  return 0;
}

function postprocessing_aborted(leave_tmps, cleanup_cmd)
{
  printf("stgcc: postprocessing aborted\n") > "/dev/stderr";
  if (leave_tmps == 0) {
    my_system(cleanup_cmd);
  }
  return 1;
}

# IN_FILE is the original input to STGCC that NOPP_ASM_FILE was generated
# from
# when PP == 1, postprocess NOPP_ASM_FILE and generate PP_ASM_FILE
# (1) gawk -f $ST_DIR/bin/asmpp.awk MODULE=IN_FILE NOPP_ASM_FILE > NOPP_ASM_FILE.pp
#     (this generates NOPP_ASM_FILE.pp and NOPP_ASM_FILE.tab.c)
# (2) gcc -I$ST_DIR/include -x c -S NOPP_ASM_FILE.tab.c
# (3) gawk -f $ST_DIR/bin/tblpp.awk $ST_DIR.tab.s > $ST_DIR.tab.s.pp
# (4) cat NOPP_ASM_FILE.pp NOPP_ASM_FILE.tab.s.pp > PP_ASM_FILE
# when PP == 0, simply copy NOPP_ASM_FILE to PP_ASM_FILE

function postprocess_file(in_file, nopp_asm_file, pp_asm_file, pp)
{
  if(show_commands)print "postprocess_file " nopp_asm_file " => " pp_asm_file;
# record the fact that PP_ASM_FILE came from IN_FILE
  postprocessed_assembly_files[in_file] = pp_asm_file;

  if (dont_execute || file_exists(nopp_asm_file)) {
    if (pp) {
      cleanup_cmd = sprintf("rm -f %s.tab.c %s.tab.s %s.tab.s.pp %s.pp",
			    nopp_asm_file, nopp_asm_file, 
			    nopp_asm_file, nopp_asm_file);
# postprocess
      cmd = sprintf("gawk -f %s/bin/%s TASK=asmpp MODULE=%s %s > %s.pp", 
		    ST_DIR, asmpp_awk, in_file, nopp_asm_file, nopp_asm_file);
      if (my_system(cmd)) {
	return postprocessing_aborted(leave_tmps, cleanup_cmd);
      }
      cmd = sprintf("%s -I%s/include -x c -S %s.tab.c", 
		    compiler, ST_DIR, nopp_asm_file);
      if (my_system(cmd)) {
	return postprocessing_aborted(leave_tmps, cleanup_cmd);
      }
      cmd = sprintf("gawk -f %s/bin/%s TASK=tblpp %s.tab.s > %s.tab.s.pp", 
		    ST_DIR, asmpp_awk, nopp_asm_file, nopp_asm_file);
      if (my_system(cmd)) {
	return postprocessing_aborted(leave_tmps, cleanup_cmd);
      }
      cmd = sprintf("cat %s.pp %s.tab.s.pp > %s", 
		    nopp_asm_file, nopp_asm_file, pp_asm_file);
      if (my_system(cmd)) {
	return postprocessing_aborted(leave_tmps, cleanup_cmd);
      }
      if (leave_tmps == 0) {
	my_system(cleanup_cmd);
      }
    } else {
# copy
      if (nopp_asm_file != pp_asm_file) {
	err = copy_file(nopp_asm_file, pp_asm_file);
	if (err) return 1;
      } else return 0;
    }
  }
  return 0;
}

function postprocess_files(\
			   nopp_asm_file, pp_asm_file)
{
  if(show_commands)print "postprocess_files";
  for (in_file in non_postprocessed_assembly_files) {
    nopp_asm_file = non_postprocessed_assembly_files[in_file];
    if (where_to_stop == stop_after_compile) {
      if (output_file) {
	pp_asm_file = output_file;
      } else {
	pp_asm_file = default_asm_output(in_file);
      }
    } else {
      if (leave_tmps) {
	pp_asm_file = default_asm_output(in_file);
      } else {
	pp_asm_file = gen_temp_file("pp", "s");
      }
    }

    err_pp = postprocess_file(in_file, nopp_asm_file, pp_asm_file, 
			      no_postprocess == 0);
    if (err_pp) return 1;
  }
  return 0;
}

# assemble ASM_FILE into OBJ_FILE. 
# LANG is either assembler-with-cpp or assembler.
# IN_FILE is the original input to STGCC that ASM_FILE was generated from
function assemble_file(lang, in_file, asm_file, obj_file,
		       opts, cmd, err)
{
  if(show_commands)print "assemble_file (" lang ") " asm_file " => " obj_file;
# record the fact that OBJ_FILE is generated from ASM_FILE
  object_files[in_file] = obj_file;
  if (dont_execute || file_exists(asm_file)) {
    if (lang == "assembler-with-cpp") {
      opts = asm_cpp_options;
    } else if (lang == "assembler") {
      opts = asm_options;
    }
    cmd = sprintf("%s %s %s -x %s -c %s -o %s",
		  compiler, opts, assemble_args, lang, asm_file, obj_file);
    err = my_system(cmd);
    return err;
  } else {
    return 0;
  }
}

# determine output file name and asemble ASM_FILE
function assemble_file1(lang, in_file, asm_file)
{
  if (where_to_stop == stop_after_assemble) {
    if (output_file) {
      obj_file = output_file;
    } else {
      obj_file = default_obj_output(in_file);
    }
  } else {
    if (leave_tmps) {
      obj_file = default_obj_output(in_file);
    } else {
      obj_file = gen_temp_file("obj", "o");
    }
  }
  assemble_file(lang, in_file, asm_file, obj_file);
}

# assemble files generated by postprocessing
# and files specified in the command line
function assemble_files(\
			i, n)
{
  if(show_commands)print "assemble_files";
# assemble all files generated by postprocessing
  for (in_file in postprocessed_assembly_files) {
    asm_file = postprocessed_assembly_files[in_file];
    assemble_file1("assembler", in_file, asm_file);
  }

# assemble all assembly files in command line
  for (l in assemble_only_langs) {
    n = n_xfiles[l];
    for (i = 0; i < n; i++) {
      asm_file = xfiles[l,i];
      assemble_file1(l, asm_file, asm_file);
    }
  }
}

# link all files
function link_files(i) # i : local variable
{
  if(show_commands)print "link_files";
# signal an error here when ST_THREAD_PKG is undefined 
# (otherwise the command line is incorrect)
  if (ST_THREAD_PKG == 0) {
    printf("Environment variable ST_THREAD_PKG not defined\n") > "/dev/stderr";
    return 1;
  }

# translate each input file into the name of .o file
  translated_args = "";
  for (i = 0; i < n_link_args; i++) {
    arg = link_args[i];
    if (arg in object_files) {
      translated_args = translated_args " " object_files[arg];
    } else {
      translated_args = translated_args " " arg;
    }
  }
  if (output_file) {
    out = " -o " output_file;
  } else {
    out = "";
  }
  cmd = sprintf("stlink --linker=%s %s %s %s -l%s -l%s %s %s", 
		linker, out, 
		translated_args, 
		stlink_common_options, libstext_a, libst_a, addlib_options,
		stlink_os_options);
  return my_system(cmd);
}

# remove files (space-separated list of file names)
function remove_files(files)
{
  if (files != "") {
    cmd = sprintf("rm -f %s", files);
    my_system(cmd);
  } else return 0;
}

function cleanup_process()
{
  if (leave_tmps == 0) {
    files = "";
    if (where_to_stop >= stop_after_compile) {
# remove all temporary .t (before postprocessing) files
      for (in_file in non_postprocessed_assembly_files) {
	f = non_postprocessed_assembly_files[in_file];
	if (f != output_file) files = files " " f;
      }
    }
    if (where_to_stop >= stop_after_assemble) {
# remove all temporary .spt files
      for (in_file in postprocessed_assembly_files) {
	f = postprocessed_assembly_files[in_file];
	if (f != output_file) files = files " " f;
      }
    }
    if (where_to_stop >= stop_after_link) {
# remove all temporary .o files
      for (in_file in object_files) {
	files = files " " object_files[in_file];
      }
    }
    remove_files(files);
  }
}

function do_compilation()
{
  if (show_commands) printf("do_compilation");
  if (where_to_stop == stop_after_cpp) {
    return preprocess_files();
  } else {
# -S | -c | nowhere-to-stop 
    if (need_compilation) {
# if any file needs compilatin, invoke the underlying C compiler
      err_cl = compile_files();
      if (err_cl) return err_cl;

# here we must have (possibly multiple) assembly files.
# postprocess these assembly files
      err_pp = postprocess_files();
      if (err_pp) return err_pp;
    }
# if -c or nowhere-to-stop, assemble assembly files
    if (where_to_stop >= stop_after_assemble) {
      err_as = assemble_files();
      if (err_as) return err_as;
    }

# if nowhere-to-stop, link files
    if (where_to_stop >= stop_after_link) {
      err_ld = link_files();
      return err_ld;
    }
  }
}

function main()
{
# check some environment variables
  if (ST_DIR == 0) {
    ST_DIR = ENVIRON["ST_DIR"];
    if (ST_DIR == 0) {
      printf("Environment variable ST_DIR not defined\n") > "/dev/stderr";
      return 1;
    }
  }
# these variables may not be defined. issue errors only when used
  if (ST_THREAD_PKG == 0) ST_THREAD_PKG = ENVIRON["ST_THREAD_PKG"];
  if (MTC_DIR == 0) MTC_DIR = ENVIRON["MTC_DIR"];
  if (SGC_DIR == 0) SGC_DIR = ENVIRON["SGC_DIR"];

# for backward compatibility (synonym for MTC_DIR)
  if (MTC_DIR == 0) MTC_DIR = ENVIRON["MTCAMP"];

  if (ST_COMPILER_PATH == 0) ST_COMPILER_PATH = ENVIRON["ST_COMPILER_PATH"];
  if ("ST_GCC_IS_PATCHED" in ENVIRON) {
    ST_GCC_IS_PATCHED = 1;
  }

  if ("ST_NEW_ASMPP" in ENVIRON) ST_NEW_ASMPP = 1;

# select compiler
  if (COMPILER) {
    compiler = COMPILER;
    linker = COMPILER;
  } else {
    compiler = "gcc";
    linker = "gcc";
  }
  if (ST_COMPILER_PATH) {
    compiler = ST_COMPILER_PATH "/" compiler;
  }

  if (EDG == 0) {
    edg = "mtcampfe";
  } else {
    edg = EDG;
  }

  asmpp_awk = "asmpp.awk";

  init_constants();
  init_variables();

  err = parse_args();
  if (err == 2) {
# a dangerous command line option was given, so we do not invoke gcc
    return 1;
  } else if (err == 1) {
# we know there are errors in the command line, but we still invoke gcc
    return invoke_error_command();
  } else {
    set_compiler_options();
    result = do_compilation();
    cleanup_process();
    return result;
  }
}

BEGIN { 
  machine_config(ENVIRON["ST_CPU_TYPE"], ENVIRON["ST_OS_TYPE"]);
}

{
  if (NR == 1) {
    r = main(); 
    exit r;
  } else {
    printf("do not use 'next'; ") > "/dev/stderr";
    printf("everything must be done in a main function\n") > "/dev/stderr";
    exit 1;
  }
}

