/* 
 * cilkcnt.c
 */

/* 
 * Copyright (c) 1999 by Kenjiro Taura, Akinori Yonezawa. All rights reserved.
 * Copyright (c) 1999 by Yoshihiro Oyama, Toshio Endo. All rights reserved.
 * Copyright (c) 1999 by Kunio Tabata. All rights reserved.
 * Copyright (c) 1999 by Mitsubishi Research Institute.  All rights reserved.
 * Copyright (c) 1999 by Information-technology Promotion Agency.  All rights reserved.
 *
 * THIS MATERIAL IS PROVIDED AS IS, WITH ABSOLUTELY NO WARRANTY EXPRESSED
 * OR IMPLIED.  ANY USE IS AT YOUR OWN RISK.
 *
 * Permission is hereby granted to use or copy this program
 * for any purpose,  provided the above notices are retained on all copies.
 * Permission to modify the code and to distribute modified code is granted,
 * provided the above notices are retained, and a notice that the code was
 * modified is included with the above copyright notice.
 */


#include <stdio.h>
#include <st.h>
#include <cilkcnt.h>

void init_cilk_sync_counter(cilk_sync_counter_t c, uslong v)
{
  c->waiting = 0;
  c->waiter_stolen = 0;
  ST_INT_LOC_INIT(&c->n_finished, 0);
  c->n_spawned = v;
}

void cilk_sync_counter_wait(cilk_sync_counter_t c)
{
  /* to grab an exclusive access to it, we must read count first
     (see cilk_sync_counter_dec) */
  if (c->waiter_stolen && !ST_INT_LOC_CHECK(&c->n_finished, c->n_spawned)) {
    int x = st_read_and_lock_int(&c->n_finished);
    /* here we have an exclusive access */
    if (x == c->n_spawned) {
      st_write_and_unlock_int(&c->n_finished, x); /* unlock */
    } else {
#if ST_DBG
      if (x < c->n_spawned) {
#endif /* ST_DBG */
	struct st_context con[1];
	con->valid = 0;
	c->waiting = con;
	st_write_and_unlock_int(&c->n_finished, x); /* unlock */
#if UWOPTIM_DBG
	fprintf(st_errout, "block\n");
#endif /* UWOPTIM_DBG */
	st_suspend_thread_n(con, 1); /* here we must grant the lock to C
					but still must protect con. */
#if ST_DBG
      } else {			/* x < c->n_spawned. something wrong */
	st_write_and_unlock_int(&c->n_finished, x); /* unlock */
	fprintf(st_errout, "%ld @ %p : n_spawned (%d) < n_finished (%d)\n",
		tls(thread_id), c, c->n_spawned, x);
	st_wg_die((void *)1);
      }
#endif
    }
  }
}

void cilk_sync_counter_finish(cilk_sync_counter_t c)
{
  if (c->waiter_stolen == 0) {
    /* we know I have the only user of this counter */
    c->n_finished.v
      = ST_INT_LOC_ATTATCH_TAG(ST_INT_LOC_STRIP_TAG(c->n_finished.v) + 1);
  } else {
    int old_count = st_read_and_lock_int(&c->n_finished);
    /* c->n_spawned may have been incremented after reading it.
       but this is no problem because in that case, c->reader == 0 */
    int new_count = old_count + 1;
    if (new_count == c->n_spawned) {
      st_context_t r = c->waiting;
      if (r) {
	c->waiting = 0;
	st_write_and_unlock_int(&c->n_finished, new_count); /* unlock */
	/* no more people should see this counter, but even if some do,
	   it will observe a negative new_count */
	if (r) st_resume_context(r);
      } else {
	st_write_and_unlock_int(&c->n_finished, new_count); /* unlock */
      }
    } else {
#if ST_DBG
      if (new_count > c->n_spawned) {
	int nsp = c->n_spawned;
	st_write_and_unlock_int(&c->n_finished, new_count); /* unlock */
	fprintf(st_errout, 
		"%ld @ %p : n_spawned (%d) < n_finished (%d)\n",
		tls(thread_id), c, nsp, new_count);
	st_wg_die((void *)1);
      }
#endif
      st_write_and_unlock_int(&c->n_finished, new_count); /* unlock */
    }
  }
}

/* called on a counter whose c->waiter_stolen != 0. 
   lock the counter and read the old n_finished */
int cilk_sync_counter_lock(cilk_sync_counter_t c)
{
  st_assert(c->waiter_stolen);
  return st_read_and_lock_int(&c->n_finished);
}

void cilk_sync_counter_finish_and_unlock(cilk_sync_counter_t c, int old_count)
{
  /* c->n_spawned may have been incremented after reading it.
     but this is no problem because in that case, c->reader == 0 */
  int new_count = old_count + 1;
  if (new_count == c->n_spawned) {
    st_context_t r = c->waiting;
    if (r) {
      c->waiting = 0;
      st_write_and_unlock_int(&c->n_finished, new_count); /* unlock */
      /* no more people should see this counter, but even if some do,
	 it will observe a negative new_count */
      if (r) st_resume_context(r);
    } else {
      st_write_and_unlock_int(&c->n_finished, new_count); /* unlock */
    }
  } else {
#if ST_DBG
    if (new_count > c->n_spawned) {
      int nsp = c->n_spawned;
      st_write_and_unlock_int(&c->n_finished, new_count); /* unlock */
      fprintf(st_errout, 
	      "%ld @ %p : n_spawned (%d) < n_finished (%d)\n",
	      tls(thread_id), c, nsp, new_count);
      st_wg_die((void *)1);
    }
#endif
    st_write_and_unlock_int(&c->n_finished, new_count); /* unlock */
  }
}

/* increase the number of required DECs after it has been created.
   it is an error to add to a counter after a wait has been performed */
void cilk_sync_counter_spawn(cilk_sync_counter_t c)
{
  c->n_spawned++;
}

#if 0				/* no longer valid */
void cilk_sync_counter_add(void *c, int a, cilk_sync_counter_t k)
{
  if (k->waiter_stolen == 0) {
    *(int*)c += a;
  } else {
    long count = st_spin_lock((long*)c); /* lock */
    count += a;            /* add */
    *(int*)c = count; /* unlock */
  }
}
#endif
