/* 
 * ralloc.c --- region-based memory allocator
 */

/* 
 * Copyright (c) 1999 by Kenjiro Taura, Akinori Yonezawa. All rights reserved.
 * Copyright (c) 1999 by Yoshihiro Oyama, Toshio Endo. All rights reserved.
 * Copyright (c) 1999 by Kunio Tabata. All rights reserved.
 * Copyright (c) 1999 by Mitsubishi Research Institute.  All rights reserved.
 * Copyright (c) 1999 by Information-technology Promotion Agency.  All rights reserved.
 *
 * THIS MATERIAL IS PROVIDED AS IS, WITH ABSOLUTELY NO WARRANTY EXPRESSED
 * OR IMPLIED.  ANY USE IS AT YOUR OWN RISK.
 *
 * Permission is hereby granted to use or copy this program
 * for any purpose,  provided the above notices are retained on all copies.
 * Permission to modify the code and to distribute modified code is granted,
 * provided the above notices are retained, and a notice that the code was
 * modified is included with the above copyright notice.
 */

#include <st.h>

/* #define REGION_DBG 1 */
#include <ralloc.h>


#define ST_REGION_DEFAULT_CHUNK_SIZE (7 * 1024) /* 63 KB */

typedef struct st_region_chunk_header
{
  struct st_region_chunk_header * next; /* next chunk */
  int size;			/* size (in bytes). include header size */
  char alloc_begin[1];		/* where the allocation is served */
} * st_region_chunk_header_t;

#define IS_ALIGNED(x) ((((long)(x)) & 7) == 0)
#define ST_REGION_BASIC_ALIGN(x) ((((long)(x))+7) & ~7)
/* if R is returned by an underlying allocater and then (R->alloc_begin) 
   is aligned by ST_REGION_BASIC_ALIGN, at most  ST_REGION_MAX_WASTE_IN_CHUNK 
   bytes is wasted at the header of the returned block of memory */
#define ST_REGION_MAX_WASTE_IN_CHUNK \
(((int)((st_region_chunk_header_t)0)->alloc_begin) + 7)
#define ST_REGION_MINIMUM_CHUNK_SIZE (ST_REGION_MAX_WASTE_IN_CHUNK + 100)

/* region information associated with each worker */
typedef struct st_region_worker_info
{
  long generation;		/* the generation when this worker has
				   performed the last allocation.
				   initially 0 */
  /* current chunk servicing allocations (an element in SMALL_CHUNKS list) */
  st_region_chunk_header_t current_chunk; 
  /* allocation pointer (marching through CURRENT_CHUNK) */
  void * p;			
  /* limit of CURRENT_CHUNK */
  void * limit;			
				
  st_region_chunk_header_t small_chunks_h; /* head of small chunks */
  st_region_chunk_header_t small_chunks_t; /* tail of small chunks */
  st_region_chunk_header_t big_chunks;	
  /* blocks of allocated memory which did not fit into a single chunk */
} * st_region_worker_info_t;

/* array of region information. its size is most of the time the number 
   of workers in the worker group in which the region is created. but 
   since workers may be added later, the size may be smaller than the actual 
   number of workers. we reallocate the array when we encounter such a 
   situation */
typedef struct st_region_worker_info_array
{
  struct st_region_worker_info_array * old; /* old version of the array */
  int size;			/* size of the array (number of elements) */
  st_region_worker_info_t a[1]; /* array (actually bigger) */
} * st_region_worker_info_array_t;

/* initialize worker-local information WI of a region.
   pre-allocate PRE_ALLOCATED_CHUNKS chunks of size CHUNK_SIZE bytes
   using UNDERLYING_ALLOC, and make them available for future allocations. */
static void st_region_worker_info_init(st_region_t rg, 
				       st_region_worker_info_t wi)
{
  st_region_chunk_header_t p;
  alloc_func_sig_t underlying_alloc = rg->underlying_alloc;
  int n_bytes = rg->n_bytes_at_first_allocation;
  /* wi->generation is zero, 
     so that it is reset in the very first allocation. */
  wi->generation = 0;
  /* big chunks are empty */
  wi->big_chunks = 0;
  /* these initializations do not matter, since it is rewritten in the 
     first allocation anyway. but we zero them just in case. */
  wi->current_chunk = 0;
  wi->p = 0;
  wi->limit = 0;
  
  p = underlying_alloc(n_bytes);
#if REGION_DBG
  printf("%ld : st_region_worker_info_init underlying_alloc(%d) ==> %p\n", 
	 tls(worker_id), n_bytes, p);
#endif
  if (p) {
    p->next = 0;
    p->size = n_bytes;
    wi->small_chunks_h = p;
    wi->small_chunks_t = p;
  } else {
    st_region_chunk_header_t h = 0, t = 0;
    int chunk_size = rg->chunk_size;
    int i;
    int n_chunks = (n_bytes + chunk_size - 1) / chunk_size;
    fprintf(st_errout, 
	    "%ld : could not pre-allocate %d bytes (still go ahead)\n",
	    tls(worker_id), n_bytes);
    for (i = 0; i < n_chunks; i++) {
      p = underlying_alloc(chunk_size);
#if REGION_DBG
      printf("%ld : st_region_worker_info_init underlying_alloc(%d) ==> %p\n", 
	     tls(worker_id), chunk_size, p);
#endif
      if (p == 0) {
	fprintf(st_errout, 
		"%ld : could not pre-allocate %dth %d bytes chunks "
		"(still go ahead)\n",
		tls(worker_id), i, chunk_size);
	break;
      }
      /* enqueue P to the queue */
      p->next = 0;
      p->size = chunk_size;
      if (h) {
	t->next = p;
      } else {
	h = p;
      }
      t = p;
    }
    wi->small_chunks_h = h;
    wi->small_chunks_t = t;
  }
}

/* ensure wa->a[idx] is a valid (non-null) pointer to st_region_worker_info */
st_region_worker_info_t 
ensure_worker_info_array_idx(st_region_t rg, st_region_worker_info_array_t wa, 
			     int idx)
{
  st_region_worker_info_array_t p;
  st_region_worker_info_t new_wi;

  /* first check if it has already been installed in an older verion of 
     the array */
  for (p = wa->old; p; p = p->old) {
    st_region_worker_info_t wi = p->a[idx];
    if (wi) {
      /* copy the pointer into the current array */
      wa->a[idx] = wi;
      return wi;
    }
  }
  /* allocate a new st_region_worker_info */
  new_wi = (st_region_worker_info_t)
    (rg->underlying_alloc(sizeof(struct st_region_worker_info)));
#if REGION_DBG
  printf("%ld : ensure_worker_info_array_idx underlying alloc(%d) ==> %p\n",
	 tls(worker_id), sizeof(struct st_region_worker_info), new_wi);
#endif
  if (new_wi == 0) {
    fprintf(st_errout, 
	    "%ld : could not allocate region_worker_info\n",
	    tls(worker_id));
    st_wg_die((void *)1);
  }
  st_region_worker_info_init(rg, new_wi);
  wa->a[idx] = new_wi;
  return new_wi;
}

/* ensure that RG->A has at least N elements large (N - 1 is an invalid index
   for RG->A) */
static st_region_worker_info_array_t 
ensure_worker_info_array_size(st_region_t rg_, int n)
{
  struct st_region volatile * rg = (struct st_region volatile *)rg_;
  st_region_worker_info_array_t wa;
  st_region_worker_info_array_t new_wa = 0;
  int new_sz = (int)(long)(((st_region_worker_info_array_t)0)->a + n);

  while (1) {
    int i;
    int old_size;
    wa = rg->worker_array;
    if (wa && n < wa->size) {
      /* somebody has expaneded the array. I do not have to do anything more */
      if (new_wa) {
#if REGION_DBG
	printf("%ld : ensure_worker_info_array_size underlying free %p\n", 
	       tls(worker_id), new_wa);
#endif
	rg->underlying_free((void *)new_wa);
      }
      new_wa = wa;
      break;
    }
    /* allocate a new array if this is the first iteration */
    if (new_wa == 0) {
      new_wa = (st_region_worker_info_array_t)(rg->underlying_alloc(new_sz));
#if REGION_DBG
      printf("%ld : ensure_worker_info_array_size underlying_alloc(%d) ==> %p\n", 
	     tls(worker_id), new_sz, new_wa);
#endif
      new_wa->size = n;
    }
    /* link new_wa and old one and set size */
    new_wa->old = wa;
    /* copy the old contents and zero new entries. the old contents
       may be updated hereafter. in that case, it may happen that wa->a[idx]
       is not zero while new_wa->a[idx] is zero. such cases are handled
       in st_region_malloc.  
    */
    if (wa) {
      /* copy the old contents */
      old_size = wa->size;
      for (i = 0; i < old_size; i++) {
	new_wa->a[i] = wa->a[i];
      }
    } else {
      old_size = 0;
    }
    /* zero new entries */
    for (i = old_size; i < n; i++) {
      new_wa->a[i] = 0;
    }
    /* try to swing the pointer to NEW_WA, if no other updates have 
       occurred in between */
    st_read_and_lock_int(&rg_->worker_array_update_lock);
    MEMBAR_READ_READ();
    if (rg->worker_array == wa) {
      rg->worker_array = new_wa;
      MEMBAR_WRITE_WRITE();
      st_write_and_unlock_int(&rg_->worker_array_update_lock, 0); /* unlock */
      break;
    } else {
      st_write_and_unlock_int(&rg_->worker_array_update_lock, 0); /* unlock */
    }
  }
  st_assert(rg->worker_array);
  return new_wa;
}

/* make region of chunk size = CHUNK_SIZE, underlying malloc/free =
   UNDERLYING_ALLOC/UNDERLYING_FREE. N_CHUNKS_IN_FIRST_ALLOCATION
   specifies how many chunks are obtained from the underlying
   allocater on the first allocation using this region.  PAO specifies
   when these chunks are allocated: when PAO ==
   st_region_pre_allocate_option_none, they are allocated on demand
   when each worker performs the first allocation.  when PAO ==
   st_region_pre_allocate_option_self, they are pre-allocated for the
   calling worker. other workers allocate them upon first allocation.
   when PAO == st_region_pre_allocate_option_all, the calling worker
   pre-allocates them for all the workers. this option is likely to have
   a (negative) side-effects on DSM machines; the first pages of these 
   chunks are touched by the calling worker.

   to make st_make_region_aux cheaper, PAO ==
   st_region_pre_allocate_option_none is useful. in this case,
   st_make_region_aux is essentially a malloc + trivial initializations. 
   There may contentions when the region is growing, however. to make
   subsequent allocations faster, give a large value to
   N_CHUNKS_IN_FIRST_ALLOCATION and give
   st_region_pre_allocate_option_all to PAO. */

st_region_t st_make_region_aux(int chunk_size, 
			       int initial_size,
			       st_region_pre_allocate_option_t pao,
			       alloc_func_sig_t underlying_alloc,
			       free_func_sig_t underlying_free)
{
  st_region_t rg;
  /* fix chunk_size, if it is too small */
#if REGION_DBG
  printf("%ld : make_region_aux(chunk_size=%d, initial_size=%d, option=%d, ..)\n",
	 tls(worker_id), chunk_size, initial_size, pao);
#endif

#if ST_DBG
  if (chunk_size < ST_REGION_MINIMUM_CHUNK_SIZE) {
    fprintf(st_errout, 
	    "%ld : warning : too small chunk_size "
	    "%d bytes fixed --> %d bytes\n",
	    tls(worker_id), chunk_size, ST_REGION_MINIMUM_CHUNK_SIZE);
    chunk_size = ST_REGION_MINIMUM_CHUNK_SIZE;
  }
#endif
  rg = (st_region_t)underlying_alloc(sizeof(struct st_region));
#if ST_DBG
  if (rg == 0) {
    fprintf(st_errout, 
	    "%ld : cannot allocate a region (%ld bytes)\n", 
	    tls(worker_id), (long)sizeof(struct st_region));
    st_wg_die((void *)1);
  }
#endif
#if REGION_DBG
  printf("%ld : make_region_aux underlying_alloc(%d) ==> %p\n",
	 tls(worker_id), sizeof(struct st_region), rg);
#endif


  rg->chunk_size = chunk_size;
  ST_INT_LOC_INIT(&rg->generation, 1);
  rg->underlying_alloc = underlying_alloc;
  rg->underlying_free = underlying_free;
  rg->worker_array = 0;
  ST_INT_LOC_INIT(&rg->worker_array_update_lock, 0);
  if (initial_size > 0) {
    long nw = st_n_current_workers();
    rg->n_bytes_at_first_allocation = (initial_size + nw - 1) / nw;
  } else {
    rg->n_bytes_at_first_allocation = chunk_size;
  }

  /* pre-allocate some chunks */
  if (pao >= st_region_pre_allocate_option_self) {
    long nw = st_n_current_workers();
    st_region_worker_info_array_t wa = ensure_worker_info_array_size(rg, nw);
    if (pao >= st_region_pre_allocate_option_all) {
      int i;
      for (i = 0; i < nw; i++) {
	ensure_worker_info_array_idx(rg, wa, i);
      }
    } else {
      ensure_worker_info_array_idx(rg, wa, tls(worker_id));
    }
  }

  return rg;
}

/* make a region which is likely to be used intensively by many workers */
st_region_t 
st_make_region_highly_concurrent_1(int init_size,
				   alloc_func_sig_t underlying_alloc,
				   free_func_sig_t underlying_free)
{
  st_region_t rg =
    st_make_region_aux(ST_REGION_DEFAULT_CHUNK_SIZE, 
		       init_size, 
		       st_region_pre_allocate_option_all,
		       underlying_alloc, underlying_free);
  return rg;
}

/* make a region which is likely to be used only by this worker */
st_region_t 
st_make_region_likely_local_1(int init_size,
			      alloc_func_sig_t underlying_alloc,
			      free_func_sig_t underlying_free)
{
  st_region_t rg =
    st_make_region_aux(ST_REGION_DEFAULT_CHUNK_SIZE, 
		       ST_REGION_DEFAULT_CHUNK_SIZE * st_n_current_workers(),
		       st_region_pre_allocate_option_self,
		       underlying_alloc, underlying_free);
  return rg;
}


/* allocate the next small chunk. first check if it is available in 
   SMALL_CHUNKS list. if it is not, ask the underlying malloc */
static void alloc_small_chunk(st_region_t rg, st_region_worker_info_t wi)
{
  st_region_chunk_header_t n;
  int req_size = rg->chunk_size;
  if (wi->current_chunk == 0) {
    /* the list is empty. request a new chunk to the underlying allocator
       and make it the element of the list */
    n = (st_region_chunk_header_t)(rg->underlying_alloc(req_size));
    if (n == 0) {
      fprintf(st_errout, "%ld : st_region_alloc out of memory\n", 
	      tls(worker_id));
      st_wg_die((void *)1);
    }
#if REGION_DBG
    printf("%ld : alloc_small_chunk underlying_alloc(%d) ==> %p\n",
	   tls(worker_id), req_size, n);
#endif
    n->next = 0;
    n->size = req_size;
    st_assert(wi->small_chunks_h == 0);
    st_assert(wi->small_chunks_t == 0);
    wi->small_chunks_h = n;
    wi->small_chunks_t = n;
  } else if (wi->current_chunk->next == 0) {
    /* we ran out of the list. request a new chunk to the underlying 
       allocator and append it to the end of the list */
    n = (st_region_chunk_header_t)(rg->underlying_alloc(req_size));
    if (n == 0) {
      fprintf(st_errout, "%ld : st_region_alloc out of memory\n", 
	      tls(worker_id));
      st_wg_die((void *)1);
    }
#if REGION_DBG
    printf("%ld : alloc_small_chunk underlying_alloc(%d) ==> %p\n",
	   tls(worker_id), req_size, n);
#endif
    /* enqueue N to SMALL_CHUNKS list */
    n->next = 0;
    n->size = req_size;
    if (wi->small_chunks_h) {
      wi->small_chunks_t->next = n;
    } else {
      wi->small_chunks_h = n;
    }
    wi->small_chunks_t = n;
  } else {
    n = wi->current_chunk->next;
  }
  /* setup allocation/limit pointers */
  wi->current_chunk = n;
  wi->p = (void *)ST_REGION_BASIC_ALIGN((void *)n->alloc_begin);
  wi->limit = (void *)n + n->size;
#if REGION_DBG
  printf("%ld : alloc_small_chunk : p = %p, limit = %p\n", 
	 tls(worker_id), wi->p, wi->limit);
#endif
}

/* allocate a chunk is SZ bytes (excluding chunk header) using the underlying
   malloc. */
static void * alloc_big_chunk(int sz, st_region_t rg, 
			      st_region_worker_info_t wi)
{
  int real_sz 
    = (int)ST_REGION_BASIC_ALIGN(sz + sizeof(struct st_region_chunk_header));
  int _ = fprintf(st_errout, "%ld : get a big chunk (%d bytes)\n", 
		  tls(worker_id), real_sz);
  /* call the underlying allocator */
  st_region_chunk_header_t 
    h = (st_region_chunk_header_t)(rg->underlying_alloc(real_sz));
  void * p = (void *)ST_REGION_BASIC_ALIGN((void *)h->alloc_begin);
#if REGION_DBG
  printf("%ld : alloc_big_chunk underlying_alloc(%d) ==> %p\n",
	 tls(worker_id), real_sz, h);
#endif

  if (h == 0) {
    fprintf(st_errout, "%ld : st_region_alloc(%d) out of memory\n", 
	    tls(worker_id), sz);
    st_wg_die((void *)1);
  }
  /* push H on top of BIG_CHUNKS list */
  h->next = wi->big_chunks;
  h->size = real_sz;
  wi->big_chunks = h;
  return p;
}

/* this worker has not seen the last reset_region, thus information in
   this worker (allocatoin pointers etc.) is obsolete. also guarantee
   that at least one small chunk is available for allocation */
static void reset_region_worker_info(st_region_t rg, 
				     st_region_worker_info_t wi)
{
  long my_generation = wi->generation;
  long world_generation = st_read_long(&rg->generation);
  st_region_chunk_header_t sc, bc;
  st_assert(my_generation != world_generation);

  /* record the fact that I have seen the last RESET_REGION(RG) */
  wi->generation = world_generation;

  /* free all big chunks */
  bc = wi->big_chunks;
  while (bc) {
    st_region_chunk_header_t nc = bc->next;
    rg->underlying_free((void *)bc);
    bc = nc;
  }
  wi->big_chunks = 0;

  /* make all small chunks available for allocation */
  sc = wi->small_chunks_h;
  wi->current_chunk = sc;
  if (sc) {
    /* set the pointer to the head of the list */
    /* set the allocation pointer and the limit pointer */
    wi->p = (void *)ST_REGION_BASIC_ALIGN((void *)sc->alloc_begin);
    wi->limit = (void *)sc + sc->size;
  } else {
    /* guarantee that we have at least one small chunk available */
    alloc_small_chunk(rg, wi);
  }
}

/* allocate SZ bytes from region RG */
void * st_region_malloc(int sz, st_region_t rg)
{
#if REGION_DBG
  int _ = printf("%ld : region_malloc %d\n", tls(worker_id), sz);
#endif
  long widx = tls(worker_id);
  st_region_worker_info_array_t wa = rg->worker_array;
  st_region_worker_info_t wi;
  long my_gen, world_gen;
  void * p;
  if (wa == 0 || widx >= wa->size) {
    /* the array turned out to be small. expand the array. */
    int nw = st_n_current_workers();
    wa = ensure_worker_info_array_size(rg, nw);
    st_assert(wa && widx < wa->size);
  }
  /* the array is big enough to hold my worker index (widx). */
  wi = wa->a[widx];
  if (wi == 0) {
    wi = ensure_worker_info_array_idx(rg, wa, widx);
  }
  my_gen = wi->generation;
  world_gen = st_read_long(&rg->generation);
  if (my_gen != world_gen) {
    /* this worker has not seen the last st_reset_region. so we reset 
       my worker info now. (if world_gen == ST_LOCKED, the world_gen is 
       being updated. i.e., somebody is resetting the region) */
    reset_region_worker_info(rg, wi);
  }
  sz = (int)ST_REGION_BASIC_ALIGN(sz);
  if (wi->p + sz < wi->limit) {
    /* the allocation can be served from the current chunk */
    p = wi->p;
    wi->p = p + sz;
  } else if (sz + ST_REGION_MAX_WASTE_IN_CHUNK <= rg->chunk_size) {
    /* the allocation fits in a single chunk, so we allocate a single chunk */
    alloc_small_chunk(rg, wi);
    p = wi->p;
    wi->p = p + sz;
  } else {
    /* the allocation does not fit in a single chunk. we search for 
       big_allocs list */
    p = alloc_big_chunk(sz, rg, wi);
  }
  st_assert(IS_ALIGNED(p));
#if REGION_DBG
  printf("%ld : region_malloc ==> %p\n", tls(worker_id), p);
#endif
  return p;
}

void * st_region_calloc(int nelem, int elsize, st_region_t rg)
{
  return st_region_malloc(nelem * elsize, rg);
}

/* abandon a worker-local chunks in region RG */
static void 
abandon_region_worker_info(st_region_t rg, st_region_worker_info_t wi)
{
  st_region_chunk_header_t sc = wi->small_chunks_h;
  st_region_chunk_header_t bc = wi->big_chunks;
  /* free all small chunks */
  while (sc) {
    st_region_chunk_header_t nc = sc->next;
    rg->underlying_free((void *)sc);
    sc = nc;
  }
  /* free all small chunks */
  while (bc) {
    st_region_chunk_header_t nc = bc->next;
    rg->underlying_free((void *)bc);
    bc = nc;
  }
}

/* abandon all chunks in a region and give them to the underlying 
   memory allocater */
void st_abandon_region(st_region_t rg)
{
  st_region_worker_info_array_t wa = rg->worker_array;
  st_region_worker_info_array_t p;
  if (wa) {
    int n = wa->size;
    int i;
    for (i = 0; i < n; i++) {
      for (p = wa; p; p = p->old) {
	if (p->a[i]) {
	  abandon_region_worker_info(rg, p->a[i]);
	  break;
	}
      }
    }
  }
  /* free worker_info_arrays */
  p = wa;
  while (p) {
    st_region_worker_info_array_t next_wa = p->old;
    rg->underlying_free((void *)p);
    p = next_wa;
  }
  rg->underlying_free((void *)rg);
}

void st_reset_region(st_region_t rg)
{
  /* increment generation counter */
  long g = st_read_and_lock_long(&rg->generation);
  if (g + 1 == ST_LOCKED) {
    /* skip over the value misidentified as locked 
       (this happens only when the generation counter wraps around) */
    g = ST_LOCKED;
  }
  st_write_and_unlock_long(&rg->generation, g + 1); /* unlock */
  /* warn if generation number becomes zero again. it happens only when
     a regin is reset so many times.
     if a worker has not malloced for a long time and its generation
     number is still zero, the worker misses a chance to reclaim 
     allocated blocks of memory, but it is still safe. the only pssibility
     is memory leak, but it is unlikely that such a worker has allocated
     much memory */
  if (g + 1 == 0) {
    fprintf(st_errout, 
	    "%ld : warning generation counter wrapped around "
	    "(probably benign. still go ahead)\n",
	    tls(worker_id));
  }
}
