/* 
 * smllc.c
 */

/* 
 * Copyright (c) 1999 by Kenjiro Taura, Akinori Yonezawa. All rights reserved.
 * Copyright (c) 1999 by Yoshihiro Oyama, Toshio Endo. All rights reserved.
 * Copyright (c) 1999 by Kunio Tabata. All rights reserved.
 * Copyright (c) 1999 by Mitsubishi Research Institute.  All rights reserved.
 * Copyright (c) 1999 by Information-technology Promotion Agency.  All rights reserved.
 *
 * THIS MATERIAL IS PROVIDED AS IS, WITH ABSOLUTELY NO WARRANTY EXPRESSED
 * OR IMPLIED.  ANY USE IS AT YOUR OWN RISK.
 *
 * Permission is hereby granted to use or copy this program
 * for any purpose,  provided the above notices are retained on all copies.
 * Permission to modify the code and to distribute modified code is granted,
 * provided the above notices are retained, and a notice that the code was
 * modified is included with the above copyright notice.
 */

#include <stdio.h>
#include <st.h>
#include <smllc.h>
#include <sys/types.h>
#include <sys/stat.h>

#define PUBLIC 
#define GLOBAL
#define PRIVATE static

/* PRIVATE may conflict with the PRIVATE in sys/mman.h*/

/*
  If you want to smalloc as fast as possible, Uncoment the followings and
  Compile this file with the --no_postprocess.
*/
/*
  #ifndef STHREADS_NOPP
  #warning "This file is recommended to be compiled with --no_postprocess"
  #endif
 */
#warning "Do NOT use smalloc with either of falloc and SGC"

#define ACQUIRE_LOCK spin_lock_intensive
static inline long spin_lock_intensive (long volatile * a);

/* Should I separete to another files? */
#define INIT_CHUNK 4 * 64 * 1024 * 1024
#define SMALLOC_MAX_SIZE 1024 * 1024 * 128
#define ALIGNMENT 8

#define FREE_LIST_LENGTH 32
typedef struct st_heap_info
{
  void *heap_p;
  void *heap_endp;
  struct free_info *free_list[FREE_LIST_LENGTH];
  volatile long MMU_miss_lock;
} * st_heap_info_t;

/* st original heap management for smalloc */
typedef struct free_info
{
  int index;
  struct free_info *next;
} * free_info_t;

/* malloc_info */
PRIVATE struct st_heap_info _malloc_info[1];

#define shr(a) _##a

/* align for sparc */
/* pending... i have to make it for a variety of architechture */
PRIVATE size_t align(size_t size){
  if(size % ALIGNMENT)
    return size + ALIGNMENT - (size % ALIGNMENT);
  else
    return size;
}

/* init smalloc:
 *
 * We have to call this exactly once before using any other following
 * functions. It is strongly recommeded that this is called before creating
 * any workers(threads).
 */
PUBLIC void smalloc_init(int nw){
  int i;
  size_t size = INIT_CHUNK;
  /* allocate large area, which will be used sliced into small pieces */
  shr(malloc_info)->heap_p = malloc(size);
  shr(malloc_info)->heap_endp = shr(malloc_info)->heap_p + size - 1;

  /* explicitly nullfy free_list */
  for(i = 0; i<FREE_LIST_LENGTH ; i++)
    shr(malloc_info)->free_list[i]=NULL;

  /* initialize MMU lock */
  shr(malloc_info)->MMU_miss_lock = 0;
}


/* We want to implement dynamic increase of heap nere future *//*pending*/
volatile alloc_next_page(void){
    fprintf(stderr, "smalloc:Memory exhausted.\n");
//    st_error();
    exit(1);
}

/* Memory allocater of the area which nobody can free */
PUBLIC void *smalloc_fixed(size_t size){
  void *allocated;
  volatile int i;
  /* round up */
  size = align(size);
  /*
   * At first, we check the size of allocated area.
   * We want to prevent st heap from reducing quickly.
   */
  if(size > SMALLOC_MAX_SIZE) return malloc(size);/* too large size */

  /* acquire lock */
  allocated = (void *)ACQUIRE_LOCK((long*)&shr(malloc_info)->heap_p);
  /* Memory exhaustion check */
  if( allocated + size > shr(malloc_info)->heap_endp ) {

    /* release lock */
    shr(malloc_info)->heap_p = 0;
    alloc_next_page();
  }

  /* allocation & release lock */
  shr(malloc_info)->heap_p = allocated + size;

  return allocated;

}  
#define MIN_MALLOC_SIZE 8
#define SPEED_UP_CILK_APPS
#ifdef SPEED_UP_CILK_APPS
#  define EXTRA_SIZE 64
#else
#  define EXTRA_SIZE 8
#endif

/* round up size to the minimun power of 2 */
PRIVATE size_t round_up(size_t size){
  int i;
  for(i = MIN_MALLOC_SIZE; i + EXTRA_SIZE < size; i <<= 1);
  return (size_t)i;
}

/* Heap Allocator:
 *
 * There two ways to get new area. First, we try to reuse freed area by
 * checking freelist. Second, we slice the large heap which is already
 * allocated by smalloc_init.
 */
PUBLIC void *smalloc(size_t size){

  int i, index=0;
  struct free_info *allocated;
  size_t rdsize;

  /* round up and get index */
  /* get suitable index from size, which holds "index = log2(size) - 3" */
  for(i = MIN_MALLOC_SIZE; i + EXTRA_SIZE < size; i <<= 1)
    index++;
  rdsize = i;
  
  /* check free list to get mem chuncks */

  /* acquire lock */
  allocated = (struct free_info *)
    ACQUIRE_LOCK( (long*)&shr(malloc_info)->free_list[index] );

  if(allocated){

    /* allocate from free list & release lock */
    shr(malloc_info)->free_list[index] = allocated -> next;

  } else {

    /* release lock */
    shr(malloc_info)->free_list[index] = allocated;

    /* allocate from heap */
    allocated = (struct free_info *)smalloc_fixed(rdsize + EXTRA_SIZE);

    /*
      The first write to an allocated area probably will cause MMU miss(page
      fault) if the allocated size exceeds hardware memory page size.
      The contension of MMU miss leads OS mutex_lock or context switch
      involuntarily. Because the overhead of them can be large, we use spin
      lock around the first write to prevent them from occuring.
      (The overhead mentioned above is very large especially on the mips.)
    */
    /* acquire lock */
    ACQUIRE_LOCK( (long*)&shr(malloc_info)->MMU_miss_lock );
    /* the first write */
    allocated->index = index; /* put size information */
    /* release lock */
    shr(malloc_info)->MMU_miss_lock = 0;
  }

  /* hide information area BEFORE the actual allocated area */
  allocated ++;

  return (void *)allocated;
}


/*
 * Free:
 *
 * Decrement the pointer to get information 'free_info' and return
 * to the global free_list.
 */
PUBLIC void sfree(void *p){
  struct free_info *new_head, *old_head;
  int index;
  
  /* decrement */
  new_head = (struct free_info *)p - 1;

  /* holds "index = log2(Allocated Size) - 3" */
  index = new_head -> index; 

  /* acquire lock */
  old_head = (struct free_info *)
    ACQUIRE_LOCK((long *)&shr(malloc_info)->free_list[index] );

  /* return to free list & release lock */
  new_head->next = old_head;
  shr(malloc_info)->free_list[index] = new_head;

  /* we don't have to return anything. */
  return;
}

/* atomically swap *a and -1. if it is already -1, retry
   until we get a value which is not -1 */
#define LOCKED -1
long spin_lock_intensive (long volatile * a)
{
  int i, j;
  long x, ok;
  while(1){
#if HAVE_SWAP

      x = SWAP_LONG(a, LOCKED);
      if (x != LOCKED) break;

#elif HAVE_LL_SC || HAVE_CMP_AND_SWAP
      x = *a;
      if(x != LOCKED) {
	ok = CMP_AND_SWAP_LONG(a, x, LOCKED);
	if (ok) break;
      }
#else

#error what do you think should I do

#endif
  }
  return x;
}
