/* 
 * ralloc.h --- region-based allocation
 */

/* 
 * Copyright (c) 1999 by Kenjiro Taura, Akinori Yonezawa. All rights reserved.
 * Copyright (c) 1999 by Yoshihiro Oyama, Toshio Endo. All rights reserved.
 * Copyright (c) 1999 by Kunio Tabata. All rights reserved.
 * Copyright (c) 1999 by Mitsubishi Research Institute.  All rights reserved.
 * Copyright (c) 1999 by Information-technology Promotion Agency.  All rights reserved.
 *
 * THIS MATERIAL IS PROVIDED AS IS, WITH ABSOLUTELY NO WARRANTY EXPRESSED
 * OR IMPLIED.  ANY USE IS AT YOUR OWN RISK.
 *
 * Permission is hereby granted to use or copy this program
 * for any purpose,  provided the above notices are retained on all copies.
 * Permission to modify the code and to distribute modified code is granted,
 * provided the above notices are retained, and a notice that the code was
 * modified is included with the above copyright notice.
 */

#ifndef __RALLOC_H__
#define __RALLOC_H__

/* NOTE: if RG_ALLOC_FUNC and RG_FREE_FUNC are defined before including this
   file, these functions are used as underlying memory
   allocator/reclaimer functions.  they are by default malloc and free. 
   e.g.,
   #define RG_ALLOC_FUNC GC_malloc
   #define RG_FREE_FUNC GC_free
   #include <ralloc.h>
   let st_make_region() etc. use them as the underlying allocator.
   no matter which values are given to RG_ALLOC_FUNC/RG_FREE_FUNC,
   you can explicitly specify the underlying allocator on a region by
   region basis (by st_make_region_aux).

   NOTE: 
   if you include st_mm.h, you do not need to include this file explicitly.
*/

/* the primary interfaces to a region-based allocator are 

   (1) st_region rg = st_make_region(), which creates a new region.

   (2) st_region_malloc(sz, rg), which allocates SZ bytes from region
   RG.

   (3) st_reset_region(rg), which makes ALL blocks of memory allocated
   by st_region_malloc(.., rg) available for subsequent calls to
   st_region_malloc. it effectively frees ALL blocks of memory
   associated with region RG, but still keeps allocated blocks
   internally, so that subsequent allocations do not have to call the
   underlying memory allocator.

   (4) st_abondon_region(rg), which
   abondon all blocks associated with RG.  it gives all blocks back to
   the underlying memory allocator.

   Overall function is similar to malloc/free, but instead of freeing
   individual blocks of memory, you reclaim all blocks associated with
   a region en messe, by a single call to st_reset_region. Both
   st_region_malloc and st_reset_region are very cheap, while
   st_make_region and st_abondon_regoin are much more expensive. 
   A typical programming example looks like this:

   rg = st_make_region();
   while (...) {
     ...
     st_region_malloc(.., rg);
     ...
     st_region_malloc(.., rg);
     ...
     st_region_malloc(.., rg);

     st_reset_region(rg);
   }

   A region can be safely reset only when all objects associated with
   the region are no longer used. This implies that, if lifetimes of
   objects associated with a region significantly differ, objects of
   short lifetimes are retained longer than necessary. Therefore there 
   are tradeoffs between the memory requirements and allocation overhead.
 */



/* the data structure that describes a region */

#ifdef __cplusplus
extern "C" {
#endif

typedef void * (*alloc_func_sig_t)(unsigned int);
typedef void (*free_func_sig_t)(void *);

typedef struct st_region
{
  /* size of a single chunk (unit of allocation to the underlying memory 
     allocator) */
  int chunk_size;		
  int n_bytes_at_first_allocation;
  /* how many times was this region reset since its birth */
  st_long_loc_t generation;
  /* underlying allocator */
  alloc_func_sig_t underlying_alloc;
  /* underlying reclaimer */
  free_func_sig_t underlying_free;
  /* array of per worker information */
  struct st_region_worker_info_array * worker_array; 
  st_int_loc_t worker_array_update_lock;
} * st_region_t;

typedef enum st_region_pre_allocate_option
{
  st_region_pre_allocate_option_none, 
  st_region_pre_allocate_option_self, 
  st_region_pre_allocate_option_all
} st_region_pre_allocate_option_t;

st_region_t 
st_make_region_aux(int /* chunk_size */, 
		   int /* pre_allocated_chunks */,
		   st_region_pre_allocate_option_t,
		   alloc_func_sig_t, free_func_sig_t);

st_region_t 
st_make_region_highly_concurrent_1(int /* expected_allocation_size */, 
				   alloc_func_sig_t, free_func_sig_t);

st_region_t 
st_make_region_likely_local_1(int /* expected_allocation_size */, 
			      alloc_func_sig_t, free_func_sig_t);

#if !defined(RG_ALLOC_FUNC) && !defined(RG_FREE_FUNC) 
#define RG_ALLOC_FUNC malloc
#define RG_FREE_FUNC free
#elif !defined(RG_ALLOC_FUNC) || !defined(RG_FREE_FUNC)
#error "one of RG_ALLOC_FUNC/RG_FREE_FUNC is defined but the other is not"
#endif

#define st_make_region_highly_concurrent(eas) \
st_make_region_highly_concurrent_1(eas, RG_ALLOC_FUNC, RG_FREE_FUNC)
#define st_make_region_likely_local(eas) \
st_make_region_highly_concurrent_1(eas, RG_ALLOC_FUNC, RG_FREE_FUNC)
#define st_make_region() st_make_region_likely_local(0)

void * st_region_malloc(int /* sz */, st_region_t);
void * st_region_calloc(int /* nelem */, int /* elsize */, st_region_t);

void st_abondon_region(st_region_t);
void st_reset_region(st_region_t);

#ifdef __cplusplus
} /* extern "C" */
#endif

#endif /* !defined(__RALLOC_H__) */
