/* 
 * st_mm.h --- basic memory management interface for C++
 */

/* 
 * Copyright (c) 1999 by Kenjiro Taura, Akinori Yonezawa. All rights reserved.
 * Copyright (c) 1999 by Yoshihiro Oyama, Toshio Endo. All rights reserved.
 * Copyright (c) 1999 by Kunio Tabata. All rights reserved.
 * Copyright (c) 1999 by Mitsubishi Research Institute.  All rights reserved.
 * Copyright (c) 1999 by Information-technology Promotion Agency.  All rights reserved.
 *
 * THIS MATERIAL IS PROVIDED AS IS, WITH ABSOLUTELY NO WARRANTY EXPRESSED
 * OR IMPLIED.  ANY USE IS AT YOUR OWN RISK.
 *
 * Permission is hereby granted to use or copy this program
 * for any purpose,  provided the above notices are retained on all copies.
 * Permission to modify the code and to distribute modified code is granted,
 * provided the above notices are retained, and a notice that the code was
 * modified is included with the above copyright notice.
 */

#ifndef __ST_MM_H__
#define __ST_MM_H__

/* NOTE: you may set the following flags before including this file:
   (1) UNDERLYING_MM (possible values are UNDERLYING_MM_SGC_AUTO_GC, 
   UNDERLYING_MM_SGC_EXP_GC, UNDERLYING_MM_SGC_EXP_FREE, 
   UNDERLYING_MM_SGC_EXP_MALLOC. see below for their meanings).
   this specifies the allocator used by MM_ALLOC and the default new (in C++)
   to class st_mm.

   (2) RALLOC (1 or 0)
   includes ralloc.h with suitable RG_ALLOC_FUNC and RG_FREE_FUNC (see
   ralloc.h), and add region-based new to class st_mm.
 */

/* the basic functionality of this file is to define three macros
   MM_ALLOC, MM_FREE, and MM_GC_POINT according to the value of
   UNDERLYING_MM, so that client code can easily change the underlying
   memory allocator. there are four options supported:
   (1) UNDERLYING_MM_SGC_AUTO_GC : rely on the automatic garbage collector
   provided by SGC. MM_ALLOC does GC_MALLOC, and MM_FREE does nothing.
   MM_GC_POINT does nothing.
   (2) UNDERLYING_MM_SGC_EXP_GC : uses garbage collector, but explicitly
   trigger GC at a point you wish to. MM_ALLOC does GC_MALLOC, MM_FREE
   does nothing. MM_GC_POINT explicitly triggers GC.
   (3) UNDERLYING_MM_SGC_EXP_FREE : uses SGC allocator, but explicitly free
   objects. it effectively uses GC_MALLOC/GC_FREE instead of malloc/free.
   MM_ALLOC does GC_MALLOC_ATOMIC, and MM_FREE does GC_FREE. MM_GC_POINT
   does nothing. note that we use GC_MALLOC_ATOMIC to save zero-clear.
   (4) UNDERLYING_MM_MALLOC : use malloc/free.

   to use anything but UNDERLYING_MM_MALLOC, you must compile st_mm.c
   with WITH_SGC = 1.

   unless RALLOC is defined to zero, ralloc is automatically included.
   ralloc uses UNDERLYING_MM as its underlying memory allocator.

   if this file is included from C++ file, it defines class st_mm. 
   classes that inherit st_mm uses MM_ALLOC for its default new.
   if the UNDERLYING_MM is UNDERLYING_MM_SGC_xxx, it also defines a new 
   with explicit object kind specification. if RALLOC, it also
   defines a new with default region specification. */

/* use ralloc by default.
   say "#define RALLOC 0" before including this file, if you do not 
   use the region-based allocator.  */
#ifndef RALLOC
#define RALLOC 1
#endif /* !defined(RALLOC) */

/* underlying memory allocator. set UNDERLYING_MM to either of the following
   values, to affect the behaviour of the default allocator (new).
   this must be set before including this file */
#define UNDERLYING_MM_SGC_AUTO_GC 0/* automatic GC (GC_dont_gc == 1) */
#define UNDERLYING_MM_SGC_EXP_GC 1 /* explicit GC (GC_dont_gc == 1) */
#define UNDERLYING_MM_SGC_EXP_FREE 2 /* GC_MALLOC_ATOMIC + GC_FREE 
					(GC_dont_gc == 1) */
#define UNDERLYING_MM_MALLOC 3	/* malloc */

/* select the default value for UNDERLYING_MM */
#ifndef UNDERLYING_MM
#if RALLOC
/* if ralloc is used, favor explicit management */
#define UNDERLYING_MM UNDERLYING_MM_MALLOC
#else  /* RALLOC */
/* if ralloc is not used, favor automatic GC */
#define UNDERLYING_MM UNDERLYING_MM_SGC_AUTO_GC
#endif /* RALLOC */
#endif /* !defined(UNDERLYING_MM) */

/* define macro MM_ALLOC, MM_FREE, and MM_GC_POINT */
#if UNDERLYING_MM == UNDERLYING_MM_SGC_AUTO_GC 
#if 0
#warning "UNDERLYING_MM == UNDERLYING_MM_SGC_AUTO_GC"
#endif
#ifdef __cplusplus
extern "C" {
#endif
  void mm_noop(void * ptr);
#ifdef __cplusplus
}
#endif

#define UNDERLYING_MM_IS_SGC 1
#define MM_ALLOC(sz) GC_malloc_ignore_off_page(sz)
#define MM_ALLOC_FUNC GC_malloc_ignore_off_page
#define MM_FREE(p) do{}while(0)
#define MM_FREE_FUNC mm_noop
#define MM_ZERO_CLEAR(ptr, sz) do{}while(0)
#define MM_GC_POINT()
#define MM_SETUP(fd) do { GC_free_space_divisor = fd; GC_dont_gc = 0; }while(0)
#elif UNDERLYING_MM == UNDERLYING_MM_SGC_EXP_GC 
#ifdef __cplusplus
extern "C" {
#endif
  void mm_noop(void * ptr);
  void * GC_malloc_ignore_off_page_make_sure_dont_gc(size_t);
#ifdef __cplusplus
}
#endif
#define UNDERLYING_MM_IS_SGC 1
#define MM_ALLOC(sz) GC_malloc_ignore_off_page_make_sure_dont_gc(sz)
#define MM_ALLOC_FUNC GC_malloc_ignore_off_page_make_sure_dont_gc
#define MM_FREE(p) do{}while(0)
#define MM_FREE_FUNC mm_noop
#define MM_ZERO_CLEAR(ptr, sz) do{}while(0)
#define MM_GC_POINT() GC_gcollect()
#define MM_SETUP(fd) do { GC_dont_gc = 1; }while(0)
#elif UNDERLYING_MM == UNDERLYING_MM_SGC_EXP_FREE
#if 0
#warning "UNDERLYING_MM == UNDERLYING_MM_SGC_EXP_FREE"
#endif
#ifdef __cplusplus
extern "C" {
#endif
  void * GC_malloc_atomic_ignore_off_page_make_sure_dont_gc(size_t);
#ifdef __cplusplus
}
#endif
#define UNDERLYING_MM_IS_SGC 1
#define MM_ALLOC(sz) GC_malloc_atomic_ignore_off_page_make_sure_dont_gc(sz)
#define MM_ALLOC_FUNC GC_malloc_atomic_ignore_off_page_make_sure_dont_gc
#define MM_FREE(p) GC_FREE(p)
#define MM_FREE_FUNC GC_free
#define MM_ZERO_CLEAR(ptr, sz) bzero(ptr, sz)
#define MM_GC_POINT() do{}while(0)
#define MM_SETUP(fd) do { GC_dont_gc = 1; }while(0)
#elif UNDERLYING_MM == UNDERLYING_MM_MALLOC
#if 0
#warning "UNDERLYING_MM == UNDERLYING_MM_MALLOC"
#endif
#define UNDERLYING_MM_IS_SGC 0
#define MM_ALLOC(sz) malloc(sz)
#define MM_ALLOC_FUNC malloc
#define MM_FREE(p) free(p)
#define MM_FREE_FUNC free
#define MM_ZERO_CLEAR(ptr, sz) bzero(ptr, sz)
#define MM_GC_POINT() do{}while(0)
#define MM_SETUP(fd) do {}while(0)
#else
#error "bad UNDERLYING_MM"
#endif

#if 0
#if RALLOC
#warning "RALLOC is used"
#else
#warning "RALLOC is not used"
#endif
#endif

/* always include stdlib.h */
#include <stdlib.h>
#include <strings.h>

/* always include header for ralloc */
#define RG_ALLOC_FUNC MM_ALLOC_FUNC
#define RG_FREE_FUNC MM_FREE_FUNC
#include <ralloc.h>
#if RALLOC
#define MM_MAKE_REGION_AUX(cs, pa, pao, ua, uf) \
st_make_region_aux(cs, pa, pao, ua, uf)

#define MM_MAKE_REGION_HIGHLY_CONCURRENT_1(cs, ua, uf) \
     st_make_region_highly_concurrent_1(cs, ua, uf)

#define MM_MAKE_REGION_LIKELY_LOCAL_1(cs, ua, uf) \
     st_make_region_likely_local_1(cs, ua, uf)

#define MM_REGION_MALLOC(sz, rg) st_region_malloc(sz, rg)
#define MM_REGION_CALLOC(sz, rg) st_region_calloc(sz, rg)
#define MM_ABANDON_REGION(rg) st_abandon_region(rg)
#define MM_RESET_REGION(rg) st_reset_region(rg)
#define MM_REGION_FREE(ptr) do{}while(0)

#else  /* RALLOC */

#define MM_MAKE_REGION_AUX(cs, pa, pao, ua, uf) 0

#define MM_MAKE_REGION_HIGHLY_CONCURRENT_1(cs, ua, uf) 0

#define MM_MAKE_REGION_LIKELY_LOCAL_1(cs, ua, uf) 0

#define MM_REGION_MALLOC(sz, rg) MM_ALLOC(sz)
#define MM_REGION_CALLOC(ne, sz, rg) MM_ALLOC(ne * sz, rg)
#define MM_ABANDON_REGION(rg) do{}while(0)
#define MM_RESET_REGION(rg) do{}while(0)
#define MM_REGION_FREE(ptr) MM_FREE(ptr)

#endif /* RALLOC */

#define MM_MAKE_REGION_HIGHLY_CONCURRENT(eas) \
     MM_MAKE_REGION_HIGHLY_CONCURRENT_1(eas, RG_ALLOC_FUNC, RG_FREE_FUNC)

#define MM_MAKE_REGION_LIKELY_LOCAL(eas) \
     MM_MAKE_REGION_LIKELY_LOCAL_1(eas, RG_ALLOC_FUNC, RG_FREE_FUNC)

#define MM_MAKE_REGION() MM_MAKE_REGION_LIKELY_LOCAL(0)

/* and sgc.h if necessary */
#if UNDERLYING_MM_IS_SGC
#include <sgc.h>
#endif /* UNDERLYING_MM_IS_SGC */

#ifdef __cplusplus

/* C++ interface: (class st_mm) */
enum mm_obj_kind {
  MM_NORMAL_OBJ,
  MM_ATOMIC_OBJ,
  MM_UNCOLLECTABLE_OBJ,
  MM_ATOMIC_UNCOLLECTABLE_OBJ
};

#if UNDERLYING_MM == UNDERLYING_MM_SGC_AUTO_GC || UNDERLYING_MM == UNDERLYING_MM_SGC_EXP_GC
#define MM_DEFAULT_OBJ_KIND MM_NORMAL_OBJ
#define MM_DEFAULT_OBJ_PLACEMENT (MM_DEFAULT_OBJ_KIND)
#elif UNDERLYING_MM == UNDERLYING_MM_SGC_EXP_FREE
#define MM_DEFAULT_OBJ_KIND MM_ATOMIC_OBJ
#define MM_DEFAULT_OBJ_PLACEMENT (MM_DEFAULT_OBJ_KIND)
#elif UNDERLYING_MM == UNDERLYING_MM_MALLOC
#define MM_DEFAULT_OBJ_PLACEMENT
#else
#error "bad UNDERLYING_MM"
#endif

struct st_mm {
  /* the default new */
  void * operator new(size_t size) {
    return MM_ALLOC(size);
  }
  void * operator new[](size_t size) {
    return operator new(size);
  }

  /* new with an explicit kind specification */
#if UNDERLYING_MM_IS_SGC
  void * operator new(size_t size, mm_obj_kind k) {
    switch(k) {
    case MM_NORMAL_OBJ:
      return GC_MALLOC(size);
      break;
    case MM_ATOMIC_OBJ:
      return GC_MALLOC_ATOMIC(size);
      break;
    case MM_UNCOLLECTABLE_OBJ:
      return GC_MALLOC_UNCOLLECTABLE(size);
      break;
    case MM_ATOMIC_UNCOLLECTABLE_OBJ:
      /* we wish to have GC_MALLOC_ATOMIC_UNCOLLECTABLE */
      return GC_malloc_atomic_uncollectable(size);
      break;
    default:
      fprintf(st_errout, "worker %d : bad mm_obj_kind = %d\n", 
	      tls(worker_id), k);
      st_app_die(1);
      break;
    }
    return 0;
  }
  
  void * operator new[](size_t size, mm_obj_kind k) {
    return operator new(size, k);
  }
#endif /* UNDERLYING_MM_IS_SGC */
  
  /* a new with an explicit region */
  void * operator new(size_t size, st_region_t rg) {
    return MM_REGION_MALLOC(size, rg);
  }
  void * operator new[](size_t size, st_region_t rg) {
    return operator new(size, rg);
  }  

};

  /* new with an explicit kind specification */
#if UNDERLYING_MM_IS_SGC
static inline void * operator new(size_t size, mm_obj_kind k) {
  switch(k) {
  case MM_NORMAL_OBJ:
    return GC_MALLOC(size);
    break;
  case MM_ATOMIC_OBJ:
    return GC_MALLOC_ATOMIC(size);
    break;
  case MM_UNCOLLECTABLE_OBJ:
    return GC_MALLOC_UNCOLLECTABLE(size);
    break;
  case MM_ATOMIC_UNCOLLECTABLE_OBJ:
    /* we wish to have GC_MALLOC_ATOMIC_UNCOLLECTABLE */
    return malloc(size);
    break;
  default:
    fprintf(st_errout, "worker %d : bad mm_obj_kind = %d\n", 
	    tls(worker_id), k);
    st_app_die(1);
    break;
  }
  return 0;
}

static inline void * operator new[](size_t size, mm_obj_kind k) {
  return operator new(size, k);
}
#endif /* UNDERLYING_MM_IS_SGC */

  /* a new with an explicit region */
static inline void * operator new(size_t size, st_region_t rg) {
  return MM_REGION_MALLOC(size, rg);
}
static inline void * operator new[](size_t size, st_region_t rg) {
  return operator new(size, rg);
}  

#endif /* __cplusplus */

#endif /* !defined(__ST_MM_H__) */

