/* 
 * ma.c
 */

/* 
 * Copyright (c) 1999 by Kenjiro Taura, Akinori Yonezawa. All rights reserved.
 * Copyright (c) 1999 by Yoshihiro Oyama, Toshio Endo. All rights reserved.
 * Copyright (c) 1999 by Kunio Tabata. All rights reserved.
 * Copyright (c) 1999 by Mitsubishi Research Institute.  All rights reserved.
 * Copyright (c) 1999 by Information-technology Promotion Agency.  All rights reserved.
 *
 * THIS MATERIAL IS PROVIDED AS IS, WITH ABSOLUTELY NO WARRANTY EXPRESSED
 * OR IMPLIED.  ANY USE IS AT YOUR OWN RISK.
 *
 * Permission is hereby granted to use or copy this program
 * for any purpose,  provided the above notices are retained on all copies.
 * Permission to modify the code and to distribute modified code is granted,
 * provided the above notices are retained, and a notice that the code was
 * modified is included with the above copyright notice.
 */

#include <stdlib.h>		/* atol, atof */
#include <st.h>
#include "st_int.h"
#ifdef WITH_SGC
#include <sgc.h>
#endif

typedef enum st_opt_type {
    st_opt_type_long, 
    st_opt_type_double,
    st_opt_type_string,
    st_opt_type_flag
} st_opt_type;

typedef enum st_opt_status {
    st_opt_status_ok,
    st_opt_status_not_found,
    st_opt_status_missing_arg,
} st_opt_status;

/* parse the command line ARGV from ARGV[0] to ARGV[*ARGCP] to find
   OPT. if it is found and an argument is required, check if one is
   given and if it is given, parse the argument accroding to the
   specified type OT to obtain a value of correct type and write the
   result to RESULT.

   option and arguments found are removed from the command line if 
   REMOVE is 1.
*/

PRIVATE st_opt_status get_opt(long * argcp, char ** argv, 
			      char ** opts, char * default_str,
			      int take_arg, st_opt_type ot, int remove,
			      void * result)
{
  long argc = *argcp;
  
  /* look for the option string in the command line */
  
  char * arg = 0;
  int k = 0;   
  int i = 0;			/* to make gcc happy */
  while ((arg == 0) && opts[k]) {
    /* look for opts[k] in argv[0] ... argv[argc-1] */
    for (i = 0; i < argc; i++) {
      if (strcmp(argv[i], opts[k]) == 0) {
	if (take_arg) {
	  if (i + 1 < argc) {
	    arg = argv[i + 1];
	    break;
	  } else {
	    /* option string is found in the last arg */
	    return st_opt_status_missing_arg;
	  }
	} else {
	  /* do not take arg. set the flag */
	  arg = "1"; 
	  break;
	}
      }
    }
    k++;
  }
  /* when the control reaches here, 
     i == argc && ARG == 0 (option was not found), or
     i < argc && ARG refers to the specified argument
     */
  
  /* apply default argument, if it is not specified */
  if (arg == 0) {
    /* option string was not found */
    if (default_str) arg = default_str;
    else return st_opt_status_not_found;
  } 
  
  /* here ARG refers to the argument specified in the command line
     or the default argument */
  st_assert(arg);
  
  switch(ot) {
    int f;
  case st_opt_type_long: /* take integer arg */
    *((long*)result) = atol(arg);
    break;
  case st_opt_type_double: /* take float arg */
    *((double*)result) = atof(arg);
    break;
  case st_opt_type_string: /* take string arg */
    *((char**)result) = arg;
    break;
  case st_opt_type_flag:	/* just a flag */
    f = st_atoflag(arg);
    if (f == -1) {
      fprintf(st_errout,
	      "default arg for option %s should be 0 or 1\n",
	      opts[k]);
      st_app_exit(1);
    } else {
      *((int *)result) = f;
    }
    break;
  default:
    fprintf(st_errout, "invalid st_opt_type %d\n", ot);
    break;
  }
  
  /* remove processed args */
  if (remove) {
    if (i < argc) {
      /* the option was found in the command line */
      int s = (take_arg ? 2 : 1);
      /* remove argv[i] and argv[i+1] (when take_arg) 
	 or remove argv[i] (when take_arg == 0) */
      int j;
      for (j = i + s; j < argc; j++) {
	argv[j - s] = argv[j];
      }
      argv[j - s] = 0;
      * argcp = argc - s;
    }
  }
  return st_opt_status_ok;
}

/* descriptor for command line option */

#define MAX_OPT_SYNONYMS 4	/* max number of synonyms for the 
				   same option */
typedef struct cmd_line_option_descriptor
{
    char * opt_names[MAX_OPT_SYNONYMS + 1]; /* string that specifies an option */
    char * descrpition;		/* brief description for help */
    char * default_str;		/* default string (0 if it is a must) */
    int take_arg;		/* 1 if it takes an argument */
    st_opt_type type;		/* type of the parameter */
    int remove;			/* 1 if found options should be removed */
    int offs;			/* the offset of the field in st_global_option */
} * cmd_line_option_descriptor_t;


/* a macro that represents the offfset of the field within 
   the global options */

#define GLOBAL_OPT_OFFSET(name) ((int)(&((st_global_options_t)0)->name))

/* add options here.

   to add an option, 
   (1) add a field in st_global_options st.h.
   (2) write an element of the structure below

   the first field is a list of option names. the leftmost one is given
   the highest priority; if two or more are specified in the command line,
   the leftmost one is taken and others are silently ignored (useful when
   the user program uses an option for its own purpose).
   
   the second field is a simple description

   the third field is a string that refers to the default value.
   write 0 if it has no default (i.e., it is required). 

   the fourth field is 1 if the option takes an argument.
   the fifth field is the type (long, double, string, or flag)
   the sixth field is 1 if found option string and arguments (if any) should 
   be removed from the command line.
   the seventh field is the offset of the field within st_global_option.
   write GLOBAL_OPT_OFFSET(field_name) here. then global_option->field_name
   is set by PARSE_OPTION routine.
*/

/* the table of command line options and their default values */
PRIVATE struct cmd_line_option_descriptor cmd_line_options[] = 
{
  /* { { names and synonyms }, "description", "default", 
     take_arg, type, remove, offs } */
  
  /* number of workers */
  { { "--n_workers", "-nw", 0 }, "number of workers", 
      "1", 1, st_opt_type_long, 1, GLOBAL_OPT_OFFSET(n_workers) },
  
  /* thread stack size */
  { { "--stack_size", "-ss", 0 }, "thread stack size", 
      "0", 1, st_opt_type_string, 1, GLOBAL_OPT_OFFSET(stack_size) },
  
  /* task steal wait limit */
  { { "--steal_wait_limit", "-sl", 0 }, "steal_wait_limit (-1 to never cancel)", 
      "40000", 1, st_opt_type_long, 1, GLOBAL_OPT_OFFSET(steal_wait_limit) },
  
  /* task steal wait warning limit */
  { { "--steal_wait_warn_limit", 0 }, "steal_wait_warn_limit", 
      "1000000", 1, st_opt_type_long, 1, 
      GLOBAL_OPT_OFFSET(steal_wait_warn_limit) },

  /* whether we should print thread stat */
  { { "--print_toplevel_worker_stat", "-ps", 0 }, "print_toplevel_worker_stat",
    "0", 0, st_opt_type_flag, 1,
      GLOBAL_OPT_OFFSET(print_toplevel_worker_stat) },
  
  /* whether we should profile throughout the application */
  { { "--time_profile", "-tp", 0 }, "time profile",
    "0", 0, st_opt_type_flag, 1,
    GLOBAL_OPT_OFFSET(time_profile) },
  
  /* file name of the profile */
  { { "--time_profile_filename", 0 }, "time profile filename",
    "00stprof", 1, st_opt_type_string, 1,
    GLOBAL_OPT_OFFSET(time_profile_filename) },
  
  /* the resolution of the time profile */
  { { "--time_profile_resolution", 0 }, "time profile resolution",
    "100", 1, st_opt_type_long, 1,
    GLOBAL_OPT_OFFSET(time_profile_resolution) },
  
  /* the in-memory buffer size of the time profile */
  { { "--time_profile_buffer_size", 0 }, "time profile buffer size",
    "8100", 1, st_opt_type_long, 1,
    GLOBAL_OPT_OFFSET(time_profile_buffer_size) },
  
  { { "--stack_unwind_optimization", "-so", 0 }, "stack_unwind_optimization",
    "0", 1, st_opt_type_long, 1,
    GLOBAL_OPT_OFFSET(stack_unwind_optimization) },

  { { 0 } }			/* end marker */
};

PRIVATE void parse_options (long * argcp, char ** argv)
{
  cmd_line_option_descriptor_t o = cmd_line_options;
  void * dest = (void *)tss(global_options);

  while (o->opt_names[0]) {
    st_opt_status s 
      = get_opt(argcp, argv, o->opt_names, o->default_str, 
		o->take_arg, o->type, o->remove, dest + o->offs);
    switch (s) {
    case st_opt_status_ok:
      break;
    case st_opt_status_not_found:
      fprintf(st_errout, 
	      "%s: command line option %s must be specified\n", 
	      argv[0], o->opt_names[0]);
      st_app_exit(1);
      break;
    case st_opt_status_missing_arg:
      fprintf(st_errout, 
	      "%s: missing argument for option %s\n",
	      argv[0], o->opt_names[0]);
      st_app_exit(1);
      break;
    default:
      fprintf(st_errout, "%s: unknown st_opt_status value (%d)\n", 
	      argv[0], s);
      st_app_exit(1);
      break;
    }
    o++;
  }
}

/* initialize sgc */
void stf_sgc_init(int n_gc_threads)
{
  ST_CALLBACK_BEGIN();
#ifdef WITH_SGC
  SGC_attr_t attr;
  SGC_attr_init(&attr);
  attr.np_fixed = 1;
  SGC_init(n_gc_threads, &attr);

  /* initialize GC-tls of main thread */
  SGC_init_local();
#endif
  ST_CALLBACK_END();
}

void stf_sgc_finalize()
{
    /* now this function is not used. */
}

/* main and some wrappers */

PRIVATE void * st_main_wrap(void * argc_, void * argv_, void * env_, void * z)
{
  int argc = (int)(long)argc_;
  char ** argv = (char **)argv_;
  char ** env = (char **)env_; 

  int r;
  DECLARE_STACK_INV_CHECK;

  SAVE_STACK_INV();
  r = st_main(argc, argv, env);
  CHECK_STACK_INV();
  return (void *)(long)r;
}

/* number of workers specified by -nw 
   unlike n_current_workers, which returns the number of workers
   already joined to the group, this is always accurate.
   */
PUBLIC int st_n_toplevel_workers()
{
  return tss(global_options)->n_workers;
}

GLOBAL void init_proc_stack_size()
{
  long ss = parse_stack_size(tss(global_options)->stack_size);
  set_proc_stack_size(ss);
}

PRIVATE void conf_toplevel_worker_group(worker_group_conf_t c)
{
  st_global_options_t g = tss(global_options);
  stf_init_worker_group_conf(c);
  stf_worker_group_conf_set_print_stat(c, g->print_toplevel_worker_stat);
  stf_worker_group_conf_set_prof_resolution(c, g->time_profile_resolution);
  stf_worker_group_conf_set_prof_buffer_size(c, g->time_profile_buffer_size);
  stf_worker_group_conf_set_prof_filename(c, g->time_profile_filename);
  stf_worker_group_conf_set_prof_do_profile(c, g->time_profile);
}

/* main */

/* 
   for those who want to use your own `main' procedure:

   we declare main as a weak symbol so that it can be overridden.
   what the default main does is mostly command line arguments processing
   and spawning toplevel workers. 

   you are reading this comment probably because you want to have more 
   controls on when workers are spawned. StackThreads/MP provides an 
   interface to spawn a group of workers. Look at sections for 
   "Cooperating with Sequential Modules" in the user's guide.
 */

#ifdef __GNUC__ 
int main (int, char **, char **) __attribute__((weak));
#endif /* __GNUC__ */

int main(int argc_, char ** argv, char ** env)
{
  ST_CALLBACK_BEGIN();
  long argc = (long)argc_;
  /* override some values in tss(global_options) by command line arguments */
  parse_options(&argc, argv);
  /* set the process stack size (thread stacks are unaffected) */
  init_proc_stack_size();
  {
    int nw = tss(global_options)->n_workers;
    void * r;
    
    stf_sgc_init(nw);

    {
      /* set configuration of the toplevel worker group
	 according to the command line options */
      struct worker_group_conf c[1];
      conf_toplevel_worker_group(c);
      /* create a group of workers that performs 
	 st_main_wrap(argc, argv, env, 0) */
      r = become_worker_group(c, nw,
			      (worker_proc_t)st_main_wrap, 
			      (void *)argc, (void *)argv, 
			      (void *)env, 0);
    }
    
    stf_sgc_finalize();
    ST_CALLBACK_END();
    return (int)(long)r;
  }
}

