/* 
 * prof.c --- profiling
 */

/* 
 * Copyright (c) 1999 by Kenjiro Taura, Akinori Yonezawa. All rights reserved.
 * Copyright (c) 1999 by Yoshihiro Oyama, Toshio Endo. All rights reserved.
 * Copyright (c) 1999 by Kunio Tabata. All rights reserved.
 * Copyright (c) 1999 by Mitsubishi Research Institute.  All rights reserved.
 * Copyright (c) 1999 by Information-technology Promotion Agency.  All rights reserved.
 *
 * THIS MATERIAL IS PROVIDED AS IS, WITH ABSOLUTELY NO WARRANTY EXPRESSED
 * OR IMPLIED.  ANY USE IS AT YOUR OWN RISK.
 *
 * Permission is hereby granted to use or copy this program
 * for any purpose,  provided the above notices are retained on all copies.
 * Permission to modify the code and to distribute modified code is granted,
 * provided the above notices are retained, and a notice that the code was
 * modified is included with the above copyright notice.
 */

#include <strings.h>		/* bzero */
#include <st.h>
#include "st_int.h"

PRIVATE void prof_change_state(prof_state_t);

GLOBAL char * prof_state_string[] =
{
  "start",
  "setup_worker",
  "busy",
  "sleep",
  "idle",
  "switch",
  "serv_steal",
  "serv_msg",
  "steal",
  "msg",
  "delete_frame",
  "spin",
  "exiting",
  "end",
  0,				/* last */
};


PRIVATE void create_prof_text_file(char * name, prof_datum_t intervals, int n)
{
#if defined(unix) || defined(__CYGWIN32__)

  /* #warning "drive out such OS-dependent stuff!" */


  FILE * f = fopen(name, "w");
  int i;
  if (f == 0) {
    fprintf(st_errout, "%ld : couldn't create file %s\n", 
	    tls(thread_id), name);
    st_wg_die((void *)1);
  }
  for (i = 0; i < n; i++) {
    fprintf(f, "%s %u %u\n",
	    prof_state_string[intervals[i].state],
	    intervals[i].begin_time, intervals[i + 1].begin_time);
  }
  if (fclose(f) != 0) {
    fprintf(st_errout, "%ld : couldn't close %s\n", 
	    tls(thread_id), name);
    st_wg_die((void *)1);
  }

#elif defined(_WIN32)

#error "do this on win32"

#else

#error "no way to create a file"

#endif
  
}

#define MAX_FILE_NAME 128
PRIVATE void flush_prof_buffer(profile_t p)
{
  char f[MAX_FILE_NAME];
  sprintf(f, "%s.%ld.%ld", p->filename_prefix, tls(thread_id), p->n_files);
  st_assert (p->n_intervals < p->max_intervals);
#if 0
  create_file(f, (char*)p->intervals, 
	      p->n_intervals * sizeof(struct prof_datum));
#else
  /* this is very slow (we should use create_file and later convert them
     to the text. for now, we use a simpler method... */
  create_prof_text_file(f, p->intervals, p->n_intervals);
#endif
  p->n_files++;
}

PRIVATE profile_t alloc_prof_buffer(prof_state_t state)
{
  long t = st_current_time_us();
  profile_t p = (profile_t)st_malloc(sizeof(struct profile));
  if (p == 0) {
    fprintf(st_errout, "couldn't allocate profile buffer\n");
    st_app_die(1);
    return 0;
  } else {
    worker_group_t wg = tls(wg);
    long m = wg->conf.prof_conf.max_intervals;
    prof_datum_t intervals 
      = (prof_datum_t)st_malloc(sizeof(struct prof_datum) * m);
    if (intervals == 0) {
      fprintf(st_errout, 
	      "couldn't allocate profile buffer of %ld elements\n", m);
      st_app_die(1);
    }
    bzero((void*)p, sizeof(struct profile));
    bzero((void*)intervals, sizeof(struct prof_datum) * m);
    p->last_prof_time = t;
    p->current_state = state;

    intervals[0].begin_time = t;
    intervals[0].state = prof_state_start;
    p->n_intervals = 1;
    p->max_intervals = m;
    p->prof_resolution = wg->conf.prof_conf.resolution;
    p->filename_prefix = wg->conf.prof_conf.filename_prefix;
    p->n_files = 0;
    p->intervals = intervals;

    /* we call current_time_use also here to count the time to setup things */
    intervals[1].begin_time = st_current_time_us();
    return p;
  }
}

PRIVATE void prof_change_state(prof_state_t new_state)
{
  if (ST_INT_LOC_CHECK(&tls(wg)->profiling, profiling_stopping)) {
    /* we are not profiling */
    return;
  } else if (ST_INT_LOC_CHECK(&tls(wg)->profiling, profiling_ending)) {
    return;
    /* we are not profiling */
  } else if (tls(profile) && tls(profile)->current_state == new_state) {
    /* the new state is the same as the current_state */
    return;
  } else {
    profiling_state_t ps = (profiling_state_t)st_read_int(&tls(wg)->profiling);
    profile_t p = tls(profile); 
    long t;

    if (ps != profiling_doing) return;

    if (p == 0) {
      p = alloc_prof_buffer(new_state);
      tls(profile) = p;
    } else {
      long n = p->n_intervals;
      t = st_current_time_us();
      /* we are in P->CURRENT_STATE between P->LAST_PROF_TIME and T */
      p->this_interval_prof[p->current_state] += (t - p->last_prof_time);
      p->total_time_prof[p->current_state] += (t - p->last_prof_time);
      /* we begin new_state from t */
      p->last_prof_time = t;
      p->current_state = new_state;
      /* check if we have experienced a long enough interval */
      if (t - p->intervals[n].begin_time > p->prof_resolution
	  || new_state == prof_state_end) {
	/* the duration of interval became longer than P->PROF_RESOLUTION.
	   now calculate the dominating state of this interval and 
	   check if it is the same as the last interval */
	prof_state_t dom_s = 0;	/* dominating state */
	{
	  long * tip = p->this_interval_prof;
	  long dom_s_length = tip[0]; /* its length */
	  prof_state_t s;
	  for (s = 1; s < prof_state_last; s++) {
	    if (tip[s] > dom_s_length) {
	      dom_s_length = tip[s];
	      dom_s = s;
	    }
	  }
	  /* prepare for the next interval (clear histgram and record
	     the start time */
	  for (s = 0; s < prof_state_last; s++) tip[s] = 0;
	}
	/* this interval is represented by state DOM_S. now check if it is
	   different from the last period */
	if (dom_s != p->intervals[n - 1].state 
	    || new_state == prof_state_end) {
	  /* the last interval was a different state. so we end the last state
	     and begins a new state. */

	  p->intervals[n].state = dom_s;
	  if (n + 1 < p->max_intervals) {
	    /* the buffer has a room to record the new state */
	    p->intervals[n + 1].begin_time = t;
	    p->n_intervals = n + 1;
	  } else {
	    /* we must flush the buffer in secondary storage */
	    fprintf(st_errout, "%ld : flush profile buffer into file\n", 
		    tls(thread_id));
	    /* flush the buffer elements intervals[0]...intervals[n-1] */
	    flush_prof_buffer(p);
	    /* move the last element to the first */ 
	    p->intervals[0] = p->intervals[n];
	    p->intervals[1].begin_time = t;
	    p->n_intervals = 1;
	  } /* if (n + 1 < p->max_intervals) */
	} else {
	  /* the interval that has just ended turned out to be the same
	     state as the last interval. so we throw away the record for
	     the interval just ended. we open an interval that starts now. */
	  p->intervals[n].begin_time = t;
	} /* if (dom_s != p->intervals[n].state) */
      } /* if (t - p->intervals[n].begin_time > p->prof_resolution) */
    } /* if (p == 0) */
  } /* if (tls(wg)->profiling == profiling_stopping) */
}

GLOBAL void st_prof_setup_worker()
{ 
  prof_change_state(prof_state_setup_worker); 
}

GLOBAL void st_prof_busy()
{ 
  prof_change_state(prof_state_busy); 
}

GLOBAL void st_prof_sleep()
{ 
  prof_change_state(prof_state_sleep); 
}

GLOBAL void st_prof_idle()
{ 
  prof_change_state(prof_state_idle); 
}

GLOBAL void st_prof_switch() 
{ 
  prof_change_state(prof_state_switch); 
}

GLOBAL void st_prof_serv_steal() 
{ 
  prof_change_state(prof_state_serv_steal); 
}

GLOBAL void st_prof_serv_msg() 
{ 
  prof_change_state(prof_state_serv_msg); 
}

GLOBAL void st_prof_steal() 
{ 
  prof_change_state(prof_state_steal); 
}

GLOBAL void st_prof_msg() 
{ 
  prof_change_state(prof_state_msg); 
}

GLOBAL void st_prof_delete_frame() 
{ 
  prof_change_state(prof_state_delete_frame); 
}

GLOBAL void st_prof_spin() 
{ 
  prof_change_state(prof_state_spin); 
}

GLOBAL void st_prof_exiting()
{ 
  prof_change_state(prof_state_exiting); 
}

/* f : filename prefix 
   res : resolution 
   m : size of the buffer */

GLOBAL void st_config_profile_1(worker_group_t wg, st_prof_conf_t pc)
{
  int x = st_read_and_lock_int(&wg->profiling);
  if (x == (int)profiling_stopping) {
    /* set all the configurations */
    wg->conf.prof_conf.do_profile = pc->do_profile;
    wg->conf.prof_conf.resolution = pc->resolution;
    wg->conf.prof_conf.max_intervals = pc->max_intervals;
    wg->conf.prof_conf.filename_prefix = pc->filename_prefix;
    /* unlock */
    st_write_and_unlock_int(&wg->profiling, (int)profiling_stopping); 
  } else {
    st_write_and_unlock_int(&wg->profiling, x); 
    fprintf(st_errout, 
	    "%ld : st_config_profile called when we are already profiling\n",
	    tls(thread_id));
    st_app_die(1);
  }
  MEMBAR_WRITE_WRITE();
}

PUBLIC void st_config_profile(st_prof_conf_t pc)
{
  st_config_profile_1(tls(wg), pc);
}

GLOBAL void st_config_profile_resolution_1(worker_group_t wg, int res)
{
  int x = st_read_and_lock_int(&wg->profiling);
  if (x == (int)profiling_stopping) {
    /* set all the configurations */
    wg->conf.prof_conf.resolution = res;
    st_write_and_unlock_int(&wg->profiling, profiling_stopping); /* unlock */
  } else {
    st_write_and_unlock_int(&wg->profiling, x); /* unlock */
    fprintf(st_errout, 
	    "%ld : st_config_profile_resolution_1 called when we are already profiling\n",
	    tls(thread_id));
    st_app_die(1);
  }
  MEMBAR_WRITE_WRITE();
}

PUBLIC void st_config_profile_resolution(int res)
{
  st_config_profile_resolution_1(tls(wg), res);
}

GLOBAL void st_config_profile_max_intervals_1(worker_group_t wg, int mi)
{
  int x = st_read_and_lock_int(&wg->profiling);
  if (x == (int)profiling_stopping) {
    /* set all the configurations */
    wg->conf.prof_conf.max_intervals = mi;
    st_write_and_unlock_int(&wg->profiling, profiling_stopping); /* unlock */
  } else {
    st_write_and_unlock_int(&wg->profiling, x);	/* unlock */
    fprintf(st_errout, 
	    "%ld : st_config_profile_resolution_1 called when we are already profiling\n",
	    tls(thread_id));
    st_app_die(1);
  }
  MEMBAR_WRITE_WRITE();
}

PUBLIC void st_config_profile_max_intervals(int mi)
{
  st_config_profile_max_intervals_1(tls(wg), mi);
}

GLOBAL void st_config_profile_filename_prefix_1(worker_group_t wg, char * f)
{
  int x = st_read_and_lock_int(&wg->profiling);
  if (x == (int)profiling_stopping) {
    /* set all the configurations */
    wg->conf.prof_conf.filename_prefix = f;
    st_write_and_unlock_int(&wg->profiling, profiling_stopping); /* unlock */
  } else {
    st_write_and_unlock_int(&wg->profiling, x); /* unlock */
    fprintf(st_errout, 
	    "%ld : st_config_profile_resolution_1 called when we are already profiling\n",
	    tls(thread_id));
    st_app_die(1);
  }
  MEMBAR_WRITE_WRITE();
}

PUBLIC void st_config_profile_filename_prefix(char * f)
{
  st_config_profile_filename_prefix_1(tls(wg), f);
}

GLOBAL void st_begin_profile_1(worker_group_t wg)
{
  int x;
  st_assert(wg->conf.prof_conf.filename_prefix);
  x = st_read_and_lock_int(&wg->profiling);
  if (x == (int)profiling_stopping) {
    st_write_and_unlock_int(&wg->profiling, profiling_doing); /* unlock */
  } else {
    st_write_and_unlock_int(&wg->profiling, x); /* unlock */
    fprintf(st_errout, 
	    "%ld : tried to begin profile when we are already profiling\n",
	    tls(thread_id));
    st_app_die(1);
  }
  prof_change_state(prof_state_busy);
}

PUBLIC void st_begin_profile()
{
  st_begin_profile_1(tls(wg));
}

typedef struct sec_and_usec
{
  long sec;
  long usec;
} sec_and_usec;

typedef struct end_profile_msg
{
  /* pointer of an array to which each worker accumulates its 
     total_time_prof (in milliseconds) */
  sec_and_usec * total_time_prof_accum;
  int reply;
} * end_profile_msg_t;

PRIVATE void end_profile_msg_handler(void * a)
{
  profile_t p = tls(profile);
  end_profile_msg_t m = (end_profile_msg_t)a;

  prof_change_state(prof_state_end);
  if (p) {
    prof_state_t s;
    /* accumulate total_time_prof */
    for (s = 0; s < prof_state_last; s++) {
      long us = p->total_time_prof[s];
      m->total_time_prof_accum[s].sec += (us / 1000000);
      m->total_time_prof_accum[s].usec += (us % 1000000);
      if (m->total_time_prof_accum[s].usec > 1000000) {
	m->total_time_prof_accum[s].sec++;
	m->total_time_prof_accum[s].usec -= 1000000;
      }
    }
  }
  m->reply = 1;
  if (p) {
    flush_prof_buffer(p);
    st_free((void*)p);
  }

  tls(profile) = 0;
}

/* send a message that askes the worker W to end profiling.
   return 0 if msg is successfully sent and acknowledged.
   return -1 otherwise */
PRIVATE int send_end_profile(workers_list_t w, 
			      sec_and_usec * total_time_prof_accum)
{
  struct end_profile_msg m_[1];
  struct end_profile_msg volatile * m 
    = (struct end_profile_msg volatile *)m_;
  struct worker_msg wm[1];

  m->total_time_prof_accum = total_time_prof_accum;
  m->reply = 0;
  if (st_send_worker_generic_msg(w, wm, end_profile_msg_handler, 
				 (void*)m) == 0) {
    while (m->reply == 0) {
      st_respond_to_worker_msg_sys(0);
      st_assert(tls(profile));
      st_assert(tls(profile)->current_state == prof_state_end);
    }
    return 0;
  } else {
    return -1;
  }
}

GLOBAL void st_end_profile_1(worker_group_t wg)
{
  workers_list_t ws = st_current_workers_1(wg);
  workers_list_t w;
  sec_and_usec total_time_prof_accum[prof_state_last];
  prof_state_t s;
  sec_and_usec total;
  int x;
  /* record the end of the profile */
  prof_change_state(prof_state_end);
  x = st_read_and_lock_int(&wg->profiling);
  /* make sure we are profiling */
  if (x == (int)profiling_doing) {
    /* unlock for now (declare ending) */
    st_write_and_unlock_int(&wg->profiling, profiling_ending); /* unlock */
  } else {
    st_write_and_unlock_int(&wg->profiling, x); /* unlock */
    fprintf(st_errout, 
	    "%ld : tried to end profile when we are not profiling\n",
	    tls(thread_id));
    st_app_die(1);
  }

  for (s = 0; s < prof_state_last; s++) {
    total_time_prof_accum[s].sec = 0;
    total_time_prof_accum[s].usec = 0;
  }
  /* ask other workers to flush your buffer */
  for (w = ws; w; w = w->next) {
    if (w != tls(worker_cell)) {
      if (send_end_profile(w, total_time_prof_accum) != 0) {
#if 1
	fprintf(st_errout, 
		"%ld : warning profile data for worker %ld may be inaccurate\n",
		tls(worker_id), w->worker_id);
#endif
      }
    }
  }
  x = st_read_and_lock_int(&wg->profiling);
  /* make sure we are ending */
  if (x == (int)profiling_ending) {
    st_write_and_unlock_int(&wg->profiling, profiling_stopping); /* unlock */
  } else {
    st_write_and_unlock_int(&wg->profiling, x);	/* unlock */
    fprintf(st_errout, 
	    "%ld : somebody has changed profiling state during finishing profile\n", tls(thread_id));
    st_app_die(1);
  }

  {
    profile_t p = tls(profile);
    if (p) {
      flush_prof_buffer(p);
      st_free((void*)p);
      /* accumulate total_time_prof */
      for (s = 0; s < prof_state_last; s++) {
	long us = p->total_time_prof[s];
	total_time_prof_accum[s].sec += (us / 1000000);
	total_time_prof_accum[s].usec += (us % 1000000);
	if (total_time_prof_accum[s].usec > 1000000) {
	  total_time_prof_accum[s].sec++;
	  total_time_prof_accum[s].usec -= 1000000;
	}
      }
    }
  }
  /* print profile summary */
  total.sec = 0;
  total.usec = 0;
  fprintf(st_errout, "PROFILE SUMMARY (ms):\n");
  for (s = 0; s < prof_state_last; s++) {
    if (total_time_prof_accum[s].sec > 0 
	|| total_time_prof_accum[s].usec >= 1000) {
      fprintf(st_errout, "%s %ld ", 
	      prof_state_string[s], 
	      total_time_prof_accum[s].sec * 1000 
	      + total_time_prof_accum[s].usec / 1000);
    }
    total.sec += total_time_prof_accum[s].sec;
    total.usec += total_time_prof_accum[s].usec;
    if (total.usec > 1000000) {
      total.sec++;
      total.usec -= 1000000;
    }
  }
  fprintf(st_errout, "\n");
  fprintf(st_errout, "TOTAL: %ld ms\n", total.sec * 1000 + total.usec / 1000);

  tls(profile) = 0;
}

PUBLIC void st_end_profile()
{
  st_end_profile_1(tls(wg));
}
