/* 
 * st_asm.c --- things that must be written in assembly
 */

/* 
 * Copyright (c) 1999 by Kenjiro Taura, Akinori Yonezawa. All rights reserved.
 * Copyright (c) 1999 by Yoshihiro Oyama, Toshio Endo. All rights reserved.
 * Copyright (c) 1999 by Kunio Tabata. All rights reserved.
 * Copyright (c) 1999 by Mitsubishi Research Institute.  All rights reserved.
 * Copyright (c) 1999 by Information-technology Promotion Agency.  All rights reserved.
 *
 * THIS MATERIAL IS PROVIDED AS IS, WITH ABSOLUTELY NO WARRANTY EXPRESSED
 * OR IMPLIED.  ANY USE IS AT YOUR OWN RISK.
 *
 * Permission is hereby granted to use or copy this program
 * for any purpose,  provided the above notices are retained on all copies.
 * Permission to modify the code and to distribute modified code is granted,
 * provided the above notices are retained, and a notice that the code was
 * modified is included with the above copyright notice.
 */

/* procedures in this file will be processed by a special postprocessor
 * that removes their prologue code. to find the end of prologue easily,
 * each procedure must begin with a comment " REAL INSN BEGINS";
 * also make sure that each procedure ends with a return instruction of
 * its own, because epilogue code is no longer valid (because we remove
 * prologue). they ought to be written in assembly in the first place,
 * but we still use C to maintain them in a single file
 */

#include <st.h>
#include "st_int.h"

/* return SP of the caller */
PUBLIC void * asm_get_sp()
{
#if defined(i386)
  asm volatile ("/ REAL INSN BEGINS");
  asm volatile ("movl %esp,%eax");
  /* because i386 return instruction increments esp by 4 */
  asm volatile ("addl $4,%eax");
  asm volatile ("ret");
  asm volatile ("/ REAL INSN ENDS");

#elif defined(mips)
  asm volatile ("# REAL INSN BEGINS");
  asm volatile ("move $2,$sp");
  asm volatile ("j $31");
  asm volatile ("# REAL INSN ENDS");

#elif defined(__sparc)
  asm volatile ("! REAL INSN BEGINS");
  asm volatile ("retl");
  asm volatile ("mov %sp,%o0");
  asm volatile ("! REAL INSN ENDS");

#elif defined(__alpha)
  asm volatile ("# REAL INSN BEGINS");
  asm volatile ("bis $30,$30,$0");
  asm volatile ("ret $31,($26),1");
  asm volatile ("# REAL INSN ENDS");

#else
#error "unknown CPU"
#endif

  return 0;			/* to make gcc happy */
}

/* return FP of the caller */
PUBLIC void * asm_get_fp()
{
#if defined(i386)
  asm volatile ("/ REAL INSN BEGINS");
  asm volatile ("movl %ebp,%eax");
  asm volatile ("ret");
  asm volatile ("/ REAL INSN ENDS");

#elif defined(mips)
  asm volatile ("# REAL INSN BEGINS");
  asm volatile ("move $2,$fp");
  asm volatile ("j $31");
  asm volatile ("# REAL INSN ENDS");

#elif defined(__sparc)
  asm volatile ("! REAL INSN BEGINS");
  asm volatile ("retl");
  asm volatile ("mov %i7,%o0");
  asm volatile ("! REAL INSN ENDS");

#elif defined(__alpha)
  asm volatile ("# REAL INSN BEGINS");
  asm volatile ("bis $15,$15,$0");
  asm volatile ("ret $31,($26),1");
  asm volatile ("# REAL INSN ENDS");

#else

#error "unknown CPU"

#endif
  return 0;			/* to make gcc happy */
}

#if HAVE_GP
/* return GP of the caller */
PUBLIC void * asm_get_gp()
{
#if defined(mips)
  asm volatile ("# REAL INSN BEGINS");
  asm volatile ("move $2,$gp");
  asm volatile ("j $31");
  asm volatile ("# REAL INSN ENDS");

#elif defined(__alpha)
  asm volatile ("# REAL INSN BEGINS");
  asm volatile ("bis $gp,$gp,$0");
  asm volatile ("ret $31,($26),1");
  asm volatile ("# REAL INSN ENDS");

#else

#error "unknown CPU that has GP"

#endif
  return 0;			/* to make gcc happy */
}
#endif /* HAVE_GP */

/* extend SP by EXT */
GLOBAL void * asm_extend_stack(long ext)
{
#if defined(i386) 
  asm volatile ("/ REAL INSN BEGINS");
  asm volatile ("movl 4(%esp),%eax"); /* eax = ext */
  /* eax = ext - 4, because RA is already pushed 
     (note that the caller frees the parameter) */
  asm volatile ("subl $4,%eax"); 
  asm volatile ("movl 0(%esp),%ecx"); /* ecx = return address */
  asm volatile ("subl %eax,%esp"); /* SP -= (ext - 4) */
  asm volatile ("movl %esp,%eax"); /* result = SP */
  asm volatile ("jmp *%ecx");
  asm volatile ("/ REAL INSN ENDS");

#elif defined(mips)
  asm volatile ("# REAL INSN BEGINS");
  asm volatile ("sub $sp,$sp,$4"); /* SP -= ext */
  asm volatile ("move $2,$sp");	/* result = SP */
  asm volatile ("j $31");
  asm volatile ("# REAL INSN ENDS");

#elif defined(__sparc)
  asm volatile ("! REAL INSN BEGINS");
  asm volatile ("sub %sp,%o0,%sp"); /* sp -= ext */
  asm volatile ("retl");
  asm volatile ("mov %sp,%o0");	/* result = SP */
  asm volatile ("! REAL INSN ENDS");

#elif defined(__alpha)
  asm volatile ("# REAL INSN BEGINS");
  asm volatile ("subq $30,$16,$30"); /* sp -= ext */
  asm volatile ("bis $30,$30,$0"); /* result =  */
  asm volatile ("ret $31,($26),1");
  asm volatile ("# REAL INSN ENDS");

#else 

#error "unknown CPU"

#endif
  return 0;
}

/* set FP/SP to the given value and goto ECA. */
GLOBAL void asm_set_fp_sp_and_jmp(void * fp, void * sp, void * eca)
{
#if defined(i386)
  asm volatile ("/ REAL INSN BEGINS");
  asm volatile ("movl 4(%esp),%ebp");	/* %ebp = FP */
  asm volatile ("movl 12(%esp),%eax");	/* %eax = ECA */
  asm volatile ("movl 8(%esp),%esp");	/* %esp = SP */
  asm volatile ("jmp *%eax");	/* goto ECA */
  asm volatile ("/ REAL INSN ENDS");

#elif defined(mips)
  asm volatile ("# REAL INSN BEGINS");
  asm volatile ("move $fp,$4");	/* $fp = fp */
  asm volatile ("move $sp,$5");	/* $sp = sp */
  asm volatile ("j $6");	/* goto eca */
  asm volatile ("# REAL INSN ENDS");

#elif defined(__sparc)
  asm volatile ("! REAL INSN BEGINS");
  asm volatile ("mov %o0,%i7");	/* %i7 = fp */
  asm volatile ("jmp %o2");	/* goto eca */
  asm volatile ("mov %o1,%sp");	/* %sp = sp */
  asm volatile ("! REAL INSN ENDS");

#elif defined(__alpha)
  asm volatile ("# REAL INSN BEGINS");
  asm volatile ("bis $16,$16,$15");	/* $fp = fp */
  asm volatile ("bis $17,$17,$30");	/* $sp = sp */
  asm volatile ("bis $18,$18,$26"); /* RA must point to the returned 
				       location */
  asm volatile ("jmp $31,($26)"); /* goto eca */
  asm volatile ("# REAL INSN ENDS");

#else

#error "unknown CPU"

#endif
}

/* set FP to the given value and goto ECA 
   (a pure epilogue code of a procedure).
   set *LINKP to the return address of this procedure + DISP, so that
   the epilogue code returns to the caller of this procedure */
GLOBAL void asm_set_fp_and_jmp_link(void * fp, void * eca, 
				    void ** linkp, long disp)
{
#if defined(i386)
  asm volatile ("/ REAL INSN BEGINS");
  asm volatile ("movl 12(%esp),%eax"); /* %eax = linkp */
  asm volatile ("movl 0(%esp),%ecx"); /* %ecx = return address */
  asm volatile ("addl 16(%esp),%ecx"); /* %ecx = return address + disp */
  asm volatile ("movl %ecx,0(%eax)"); /* *linkp = return address + disp */
  asm volatile ("movl 4(%esp),%ebp");	/* %ebp = FP */
  asm volatile ("movl 8(%esp),%eax");	/* %eax = ECA */
  /* clean up stack (4 params and return address) */
  asm volatile ("addl $4,%esp"); /* cleanup return address */
  asm volatile ("jmp *%eax");	/* goto ECA */
  asm volatile ("/ REAL INSN ENDS");

#elif defined(mips)
  asm volatile ("# REAL INSN BEGINS");
  asm volatile ("add $2,$31,$7"); /* $2 = return address + disp */
  asm volatile ("sw $2,0($6)");	/* *linkp = return address + disp */
  asm volatile ("move $fp,$4");	/* $fp = fp */
  asm volatile ("j $5");	/* goto eca */
  asm volatile ("# REAL INSN ENDS");

#elif defined(__sparc)
  asm volatile ("! REAL INSN BEGINS");
  asm volatile ("add %o7,%o3,%o4"); /* $2 = return address + disp */
  asm volatile ("st %o4,[%o2]"); /* *linkp = return address + disp */
  asm volatile ("jmp %o1");	/* goto eca */
  asm volatile ("mov %o0,%i7");	/* %i7 = fp */
  asm volatile ("! REAL INSN ENDS");

#elif defined(__alpha)
  asm volatile ("# REAL INSN BEGINS");
  asm volatile ("addq $26,$19,$1"); /* $1 = return address + disp */
  asm volatile ("stq $1,0($18)"); /* *linkp = return address + disp */
  asm volatile ("bis $16,$16,$15");	/* $fp = fp */
  asm volatile ("bis $17,$17,$26"); /* RA must point to the returned 
				       location */
  asm volatile ("jmp $31,($26)"); /* goto eca */
  asm volatile ("# REAL INSN ENDS");

#else

#error "unknown CPU"

#endif
}

/* this procedure returns twice. upon called, it saves callee-save
   registers, frame pointer, and PC (of the caller) to C and returns
   zero. it later returns again when somebody calls install_context
   with the saved context. this time, it return 1 and SP points to the 
   caller of INSTALL_CONTEXT (stack top) */

/* # of parameters must be equal to asm_install_context */
GLOBAL int asm_capture_context(st_context_t c)
{
#if defined(i386)
  /* sp[0] == RA, sp[1] == C.
     note that we do not setup frame pointers for this function
     on i386, stack size is increased by four because parameter C
     and return address is pushed onto the stack. we save SP when 
     nothing is pushed */

  asm volatile ("/ REAL INSN BEGINS");
  asm volatile ("movl 4(%esp),%eax"); /* %eax = C */
  /* c->sp = %esp + 8 (return address and parameter C).
     change here when # of parameters change */
  asm volatile ("movl %esp,0(%eax)"); 
  asm volatile ("addl $8,0(%eax)");

  asm volatile ("movl %ebp,4(%eax)"); /* c->fp = %ebp */
  asm volatile ("movl 0(%esp),%ecx"); /* c->pc = RA */
  asm volatile ("movl %ecx,8(%eax)");
  asm volatile ("movl %esi,12(%eax)"); /* c->regs[0] = %esi */
  asm volatile ("movl %edi,16(%eax)"); /* c->regs[1] = %edi */
  asm volatile ("movl $0,%eax"); /* return value = 0 */
  asm volatile ("ret");
  asm volatile ("/ REAL INSN ENDS");

#elif defined(mips)
  asm volatile ("# REAL INSN BEGINS");
  asm volatile ("sw $sp,0($4)"); /* c->sp = %esp */
  asm volatile ("sw $fp,4($4)"); /* c->fp = %ebp */
  asm volatile ("sw $31,8($4)"); /* c->pc = RA */
  asm volatile ("s.d $f20,16($4)"); /* c->regs[0] = $f20 */
  asm volatile ("s.d $f22,24($4)"); /* c->regs[2] = $f22 */
  asm volatile ("s.d $f24,32($4)"); /* c->regs[4] = $f24 */
  asm volatile ("s.d $f26,40($4)"); /* c->regs[6] = $f26 */
  asm volatile ("s.d $f28,48($4)"); /* c->regs[8] = $f28 */
  asm volatile ("s.d $f30,56($4)"); /* c->regs[10] = $f30 */
  asm volatile ("sw $16,64($4)"); /* c->regs[12] = $16 */
  asm volatile ("sw $17,68($4)"); /* c->regs[13] = $17 */
  asm volatile ("sw $18,72($4)"); /* c->regs[14] = $18 */
  asm volatile ("sw $19,76($4)"); /* c->regs[15] = $19 */
  asm volatile ("sw $20,80($4)"); /* c->regs[16] = $20 */
  asm volatile ("sw $21,84($4)"); /* c->regs[17] = $21 */
  asm volatile ("sw $22,88($4)"); /* c->regs[18] = $22 */
  asm volatile ("li $2,0x00000000"); /* return value = 0 */
  asm volatile ("j $31");
  asm volatile ("# REAL INSN ENDS");

#elif defined(__sparc)
  asm volatile ("! REAL INSN BEGINS");
  asm volatile ("st %sp,[%o0]"); /* c->sp = %esp */
  asm volatile ("st %i7,[%o0+4]"); /* c->fp = %ebp */
  asm volatile ("st %o7,[%o0+8]");
  asm volatile ("std %i0,[%o0+16]"); /* c->regs[0,1] = %i0,%i1 */
  asm volatile ("std %i2,[%o0+24]"); /* c->regs[2,3] = %i2,%i3 */
  asm volatile ("std %i4,[%o0+32]"); /* c->regs[4,5] = %i4,%i5 */
  asm volatile ("std %l0,[%o0+40]"); /* c->regs[6,7] = %l0,%l1 */
  asm volatile ("std %l2,[%o0+48]"); /* c->regs[8,9] = %l2,%l3 */
  asm volatile ("std %l4,[%o0+56]"); /* c->regs[10,11] = %l4,%l5 */
  asm volatile ("st %l6,[%o0+64]"); /* c->regs[12] = %l6 */
  asm volatile ("retl");
  asm volatile ("mov %g0,%o0"); /* return value = 0 */
  asm volatile ("! REAL INSN ENDS");

#elif defined(__alpha)
  asm volatile ("# REAL INSN BEGINS");
  asm volatile ("stq $30,0($16)"); /* c->sp = %esp */
  asm volatile ("stq $15,8($16)"); /* c->fp = %ebp */
  asm volatile ("stq $26,16($16)"); /* c->pc = RA */
  asm volatile ("stt $f2,24($16)"); /* c->regs[0] = $f2 */
  asm volatile ("stt $f3,32($16)"); /* c->regs[1] = $f3 */
  asm volatile ("stt $f4,40($16)"); /* c->regs[2] = $f4 */
  asm volatile ("stt $f5,48($16)"); /* c->regs[3] = $f5 */
  asm volatile ("stt $f6,56($16)"); /* c->regs[4] = $f6 */
  asm volatile ("stt $f7,64($16)"); /* c->regs[5] = $f7 */
  asm volatile ("stt $f8,72($16)"); /* c->regs[6] = $f8 */
  asm volatile ("stt $f9,80($16)"); /* c->regs[7] = $f9 */
  asm volatile ("stq $9,88($16)"); /* c->regs[8] = $9 */
  asm volatile ("stq $10,96($16)"); /* c->regs[9] = $10 */
  asm volatile ("stq $11,104($16)"); /* c->regs[10] = $11 */
  asm volatile ("stq $12,112($16)"); /* c->regs[11] = $12 */
  asm volatile ("stq $13,120($16)"); /* c->regs[12] = $13 */
  asm volatile ("bis $31,$31,$0"); /* return value = 0 */
  asm volatile ("ret $31,($26),1");
  asm volatile ("# REAL INSN ENDS");

#else

#error "unknown CPU"

#endif
  return 0;			/* to make gcc happy */
}

/* C is a pointer to a context that has been captured by capture_context.
   this routine restores captured context, and jump to that point as if
   capture_context returns 1. */

GLOBAL int asm_install_context(st_context_t c)
{
#if defined(i386)
  /* sp[1] == C */
  asm volatile ("/ REAL INSN BEGINS");
  asm volatile ("movl 4(%esp),%eax"); /* %eax = C */
  asm volatile ("movl 4(%eax),%ebp"); /* %ebp = c->fixed_fp */
  asm volatile ("movl 8(%eax),%ecx"); /* %ecx = return address */
  asm volatile ("movl 12(%eax),%esi"); /* %esi = c->fixed_regs[0] */
  asm volatile ("movl 16(%eax),%edi"); /* %edi = c->fixed_regs[1] */
  asm volatile ("movl $1,%eax"); /* return value = 1 */
  /* cleanup return address */
  asm volatile ("addl $4,%esp"); 
  asm volatile ("jmp *%ecx");
  asm volatile ("/ REAL INSN ENDS");

#elif defined(mips)
  asm volatile ("# REAL INSN BEGINS");
  asm volatile ("lw	$fp,4($4)"); /* $fp = c->fixed_fp */
  asm volatile ("lw	$31,8($4)"); /* $ra = c->fixed_ra */
  asm volatile ("l.d	$f20,16($4)"); /* $f20 = c->regs[0] */
  asm volatile ("l.d	$f22,24($4)"); /* $f22 = c->regs[2] */
  asm volatile ("l.d	$f24,32($4)"); /* $f24 = c->regs[4] */
  asm volatile ("l.d	$f26,40($4)"); /* $f26 = c->regs[6] */
  asm volatile ("l.d	$f28,48($4)"); /* $f28 = c->regs[8] */
  asm volatile ("l.d	$f30,56($4)"); /* $f30 = c->regs[10] */
  asm volatile ("lw	$16,64($4)"); /* $16 = c->regs[12] */
  asm volatile ("lw	$17,68($4)"); /* $17 = c->regs[13] */
  asm volatile ("lw	$18,72($4)"); /* $18 = c->regs[14] */
  asm volatile ("lw	$19,76($4)"); /* $19 = c->regs[15] */
  asm volatile ("lw	$20,80($4)"); /* $20 = c->regs[16] */
  asm volatile ("lw	$21,84($4)"); /* $21 = c->regs[17] */
  asm volatile ("lw	$22,88($4)"); /* $22 = c->regs[18] */
  asm volatile ("li	$2,0x00000001"); /* return value = 0 */
  asm volatile ("j	$31");
  asm volatile ("# REAL INSN ENDS");

#elif defined(__sparc)
  asm volatile ("! REAL INSN BEGINS");
  asm volatile ("ld [%o0+4],%i7"); /* %i7 = c->fixed_fp */
  asm volatile ("ld [%o0+8],%o7"); /* %o7 = c->fixed_ra */
  asm volatile ("ldd [%o0+16],%i0"); /* %i0,%i1 = c->regs[0,1] */
  asm volatile ("ldd [%o0+24],%i2"); /* %i2,%i3 = c->regs[2,3] */
  asm volatile ("ldd [%o0+32],%i4"); /* %i4,%i5 = c->regs[4,5] */
  asm volatile ("ldd [%o0+40],%l0"); /* %l0,%l1 = c->regs[6,7] */
  asm volatile ("ldd [%o0+48],%l2"); /* %l2,%l3 = c->regs[8,9] */
  asm volatile ("ldd [%o0+56],%l4"); /* %l4,%l5 = c->regs[10,11] */
  asm volatile ("ld [%o0+64],%l6"); /* %l6 = c->regs[12] */
  asm volatile ("retl");
  asm volatile ("mov 1,%o0");
  asm volatile ("! REAL INSN ENDS");

#elif defined(__alpha)
  asm volatile ("# REAL INSN BEGINS");
  asm volatile ("ldq $15,8($16)"); /* $fp = c->fixed_fp */
  asm volatile ("ldq $26,16($16)"); /* $ra = c->fixed_ra */
  asm volatile ("ldt $f2,24($16)"); /* $f2 = c->regs[0] */
  asm volatile ("ldt $f3,32($16)"); /* $f3 = c->regs[1] */
  asm volatile ("ldt $f4,40($16)"); /* $f4 = c->regs[2] */
  asm volatile ("ldt $f5,48($16)"); /* $f5 = c->regs[3] */
  asm volatile ("ldt $f6,56($16)"); /* $f6 = c->regs[4] */
  asm volatile ("ldt $f7,64($16)"); /* $f7 = c->regs[5] */
  asm volatile ("ldt $f8,72($16)"); /* $f8 = c->regs[6] */
  asm volatile ("ldt $f9,80($16)"); /* $f9 = c->regs[7] */
  asm volatile ("ldq $9,88($16)"); /* $9 = c->regs[8] */
  asm volatile ("ldq $10,96($16)"); /* $10 = c->regs[9] */
  asm volatile ("ldq $11,104($16)"); /* $11 = c->regs[10] */
  asm volatile ("ldq $12,112($16)"); /* $12 = c->regs[11] */
  asm volatile ("ldq $13,120($16)"); /* $13 = c->regs[12] */
  asm volatile ("bis $31,1,$0"); /* return value = 0 */
  asm volatile ("jmp $31,($26)");
  asm volatile ("# REAL INSN ENDS");

#else

#error "unknown CPU"

#endif
  return 0;			/* to make gcc happy */
}


/* called immediately after restart_context is returned, at which point
   callee-save registers are invalid. reload callee-save registers from
   tls(fixed_invalid_frames). */
USED_BY_MACRO void asm_fix_callee_saves()
{
#if defined(i386)
  asm volatile ("/ REAL INSN BEGINS");
  asm volatile ("movl 4(%ebx),%eax"); /* eax = tls(fixed_invalid_frames) */
  /* esi = tls(fixed_invalid_frames)->fixed_cs_regs[0] */
  asm volatile ("movl 0(%eax),%esi"); 
  /* edi = tls(fixed_invalid_frames)->fixed_cs_regs[1] */
  asm volatile ("movl 4(%eax),%edi"); 

  asm volatile ("ret");
  asm volatile ("/ REAL INSN ENDS");

#elif defined(mips)
  asm volatile ("# REAL INSN BEGINS");

  asm volatile ("lw	$4,4($23)"); /* $4 = tls(fixed_invalid_frames) */
  /* $f20 = tls(fixed_invalid_frames)->fixed_cs_regs[0] */
  asm volatile ("l.d	$f20,0($4)"); 
  /* $f22 = tls(fixed_invalid_frames)->fixed_cs_regs[2] */
  asm volatile ("l.d	$f22,8($4)");
  /* $f24 = tls(fixed_invalid_frames)->fixed_cs_regs[4] */
  asm volatile ("l.d	$f24,16($4)");
  /* $f26 = tls(fixed_invalid_frames)->fixed_cs_regs[6] */
  asm volatile ("l.d	$f26,24($4)");
  /* $f28 = tls(fixed_invalid_frames)->fixed_cs_regs[8] */
  asm volatile ("l.d	$f28,32($4)");
  /* $f30 = tls(fixed_invalid_frames)->fixed_cs_regs[10] */
  asm volatile ("l.d	$f30,40($4)");
  /* $16 = tls(fixed_invalid_frames)->fixed_cs_regs[12] */
  asm volatile ("lw	$16,48($4)"); 
  /* $16 = tls(fixed_invalid_frames)->fixed_cs_regs[13] */
  asm volatile ("lw	$17,52($4)");
  /* $16 = tls(fixed_invalid_frames)->fixed_cs_regs[14] */
  asm volatile ("lw	$18,56($4)");
  /* $16 = tls(fixed_invalid_frames)->fixed_cs_regs[15] */
  asm volatile ("lw	$19,60($4)");
  /* $16 = tls(fixed_invalid_frames)->fixed_cs_regs[16] */
  asm volatile ("lw	$20,64($4)");
  /* $16 = tls(fixed_invalid_frames)->fixed_cs_regs[17] */
  asm volatile ("lw	$21,68($4)");
  /* $16 = tls(fixed_invalid_frames)->fixed_cs_regs[18] */
  asm volatile ("lw	$22,72($4)");
  asm volatile ("j	$31");
  asm volatile ("# REAL INSN ENDS");


#elif defined(__sparc)
  asm volatile ("! REAL INSN BEGINS");

  asm volatile ("ld [%l7+4],%o0"); /* o0 = tls(fixed_invalid_frames) */
  /* %i0,%i1 = tls(fixed_invalid_frames)->fixed_cs_regs[0,1] */
  asm volatile ("ldd [%o0],%i0"); 
  /* %i2,%i3 = tls(fixed_invalid_frames)->fixed_cs_regs[2,3] */
  asm volatile ("ldd [%o0+8],%i2");
  /* %i4,%i5 = tls(fixed_invalid_frames)->fixed_cs_regs[4,5] */
  asm volatile ("ldd [%o0+16],%i4");
  /* %l0,%l1 = tls(fixed_invalid_frames)->fixed_cs_regs[6,7] */
  asm volatile ("ldd [%o0+24],%l0");
  /* %l2,%l3 = tls(fixed_invalid_frames)->fixed_cs_regs[8,9] */
  asm volatile ("ldd [%o0+32],%l2");
  /* %l4,%l5 = tls(fixed_invalid_frames)->fixed_cs_regs[10,11] */
  asm volatile ("ldd [%o0+40],%l4");
  /* return with delay slot */
  asm volatile ("retl");
  /* %l6 = tls(fixed_invalid_frames)->fixed_cs_regs[12] */
  asm volatile ("ld [%o0+48],%l6");
  asm volatile ("! REAL INSN ENDS");

#elif defined(__alpha)
  asm volatile ("# REAL INSN BEGINS");

  /* $16 = tls(fixed_invalid_frames) */
  asm volatile ("ldq $16,8($14)"); 
  /* $f2 = tls(fixed_invalid_frames)->fixed_cs_regs[0] */
  asm volatile ("ldt $f2,0($16)");
  /* $f3 = tls(fixed_invalid_frames)->fixed_cs_regs[1] */
  asm volatile ("ldt $f3,8($16)");
  /* $f4 = tls(fixed_invalid_frames)->fixed_cs_regs[2] */
  asm volatile ("ldt $f4,16($16)");
  /* $f5 = tls(fixed_invalid_frames)->fixed_cs_regs[3] */
  asm volatile ("ldt $f5,24($16)");
  /* $f6 = tls(fixed_invalid_frames)->fixed_cs_regs[4] */
  asm volatile ("ldt $f6,32($16)");
  /* $f7 = tls(fixed_invalid_frames)->fixed_cs_regs[5] */
  asm volatile ("ldt $f7,40($16)");
  /* $f8 = tls(fixed_invalid_frames)->fixed_cs_regs[6] */
  asm volatile ("ldt $f8,48($16)");
  /* $f9 = tls(fixed_invalid_frames)->fixed_cs_regs[7] */
  asm volatile ("ldt $f9,56($16)");
  /* $9 = tls(fixed_invalid_frames)->fixed_cs_regs[8] */
  asm volatile ("ldq $9,64($16)");
  /* $10 = tls(fixed_invalid_frames)->fixed_cs_regs[9] */
  asm volatile ("ldq $10,72($16)");
  /* $11 = tls(fixed_invalid_frames)->fixed_cs_regs[10] */
  asm volatile ("ldq $11,80($16)");
  /* $12 = tls(fixed_invalid_frames)->fixed_cs_regs[11] */
  asm volatile ("ldq $12,88($16)");
  /* $13 = tls(fixed_invalid_frames)->fixed_cs_regs[12] */
  asm volatile ("ldq $13,96($16)");
  asm volatile ("ret $31,($26),1");
  asm volatile ("# REAL INSN ENDS");

#else

#error "unknown CPU"

#endif
  
}

/* save callee-save registers to CS */
GLOBAL void asm_save_callee_saves(uslong * cs)
{
#if defined(i386)
  asm volatile ("/ REAL INSN BEGINS");
  asm volatile ("movl 4(%esp),%eax"); /* %eax = cs */
  asm volatile ("movl %esi,0(%eax)"); /* %esi = cs[0] */
  asm volatile ("movl %edi,4(%eax)"); /* %edi = cs[1] */
  asm volatile ("ret");
  asm volatile ("/ REAL INSN ENDS");

#elif defined(mips)
  asm volatile ("# REAL INSN BEGINS");
  asm volatile ("s.d	$f20,0($4)"); /* cs[0] = $f20 */
  asm volatile ("s.d	$f22,8($4)"); /* cs[2] = $f22 */
  asm volatile ("s.d	$f24,16($4)"); /* cs[4] = $f24 */
  asm volatile ("s.d	$f26,24($4)"); /* cs[6] = $f26 */
  asm volatile ("s.d	$f28,32($4)"); /* cs[8] = $f28 */
  asm volatile ("s.d	$f30,40($4)"); /* cs[10] = $f30 */
  asm volatile ("sw	$16,48($4)"); /* cs[12] = $16 */
  asm volatile ("sw	$17,52($4)"); /* cs[13] = $17 */
  asm volatile ("sw	$18,56($4)"); /* cs[14] = $18 */
  asm volatile ("sw	$19,60($4)"); /* cs[15] = $19 */
  asm volatile ("sw	$20,64($4)"); /* cs[16] = $20 */
  asm volatile ("sw	$21,68($4)"); /* cs[17] = $21 */
  asm volatile ("sw	$22,72($4)"); /* cs[18] = $22 */
  asm volatile ("j	$31");
  asm volatile ("# REAL INSN ENDS");

#elif defined(__sparc)
  asm volatile ("! REAL INSN BEGINS");
  asm volatile ("std %i0,[%o0]"); /* cs[0,1] = %i0,%i1 */
  asm volatile ("std %i2,[%o0+8]"); /* cs[2,3] = %i2,%i3 */
  asm volatile ("std %i4,[%o0+16]"); /* cs[4,5] = %i4,%i5 */
  asm volatile ("std %l0,[%o0+24]"); /* cs[6,7] = %l0,%l1 */
  asm volatile ("std %l2,[%o0+32]"); /* cs[8,9] = %l2,%l3 */
  asm volatile ("std %l4,[%o0+40]"); /* cs[10,11] = %l4,%l5 */
  asm volatile ("retl");
  asm volatile ("st	%l6,[%o0+48]"); /* cs[12] = %l6 */
  asm volatile ("! REAL INSN ENDS");

#elif defined(__alpha)
  asm volatile ("# REAL INSN BEGINS");
  asm volatile ("stt $f2,0($16)"); /* cs[0] = $f2 */
  asm volatile ("stt $f3,8($16)"); /* cs[1] = $f3 */
  asm volatile ("stt $f4,16($16)"); /* cs[2] = $f4 */
  asm volatile ("stt $f5,24($16)"); /* cs[3] = $f5 */
  asm volatile ("stt $f6,32($16)"); /* cs[4] = $f6 */
  asm volatile ("stt $f7,40($16)"); /* cs[5] = $f7 */
  asm volatile ("stt $f8,48($16)"); /* cs[6] = $f8 */
  asm volatile ("stt $f9,56($16)"); /* cs[7] = $f9 */
  asm volatile ("stq $9,64($16)"); /* cs[8] = $9 */
  asm volatile ("stq $10,72($16)"); /* cs[9] = $10 */
  asm volatile ("stq $11,80($16)"); /* cs[10] = $11 */
  asm volatile ("stq $12,88($16)"); /* cs[11] = $12 */
  asm volatile ("stq $13,96($16)"); /* cs[12] = $13 */
  asm volatile ("ret $31,($26),1");
  asm volatile ("# REAL INSN ENDS");

#else

#error "unknown CPU"

#endif
}

GLOBAL void asm_restore_callee_saves(uslong * cs)
{
#if defined(i386)
  asm volatile ("/ REAL INSN BEGINS");
  asm volatile ("movl 4(%esp),%eax"); /* %eax = cs */
  asm volatile ("movl 0(%eax),%esi"); /* %esi = cs[0] */
  asm volatile ("movl 4(%eax),%edi"); /* %edi = cs[1] */
  asm volatile ("ret");
  asm volatile ("/ REAL INSN ENDS");

#elif defined(mips)
  asm volatile ("# REAL INSN BEGINS");
  asm volatile ("l.d	$f20,0($4)"); /* $f20 = cs[0] */
  asm volatile ("l.d	$f22,8($4)"); /* $f22 = cs[2] */
  asm volatile ("l.d	$f24,16($4)"); /* $f24 = cs[4] */
  asm volatile ("l.d	$f26,24($4)"); /* $f26 = cs[6] */
  asm volatile ("l.d	$f28,32($4)"); /* $f28 = cs[8] */
  asm volatile ("l.d	$f30,40($4)"); /* $f30 = cs[10] */
  asm volatile ("lw	$16,48($4)"); /* $16 = cs[12] */
  asm volatile ("lw	$17,52($4)"); /* $17 = cs[13] */
  asm volatile ("lw	$18,56($4)"); /* $18 = cs[14] */
  asm volatile ("lw	$19,60($4)"); /* $19 = cs[15] */
  asm volatile ("lw	$20,64($4)"); /* $20 = cs[16] */
  asm volatile ("lw	$21,68($4)"); /* $21 = cs[17] */
  asm volatile ("lw	$22,72($4)"); /* $22 = cs[18] */
  asm volatile ("j	$31");
  asm volatile ("# REAL INSN ENDS");

#elif defined(__sparc)
  asm volatile ("! REAL INSN BEGINS");
  asm volatile ("ldd [%o0],%i0"); /* %i0,%i1 = cs[0,1] */
  asm volatile ("ldd [%o0+8],%i2"); /* %i2,%i3 = cs[2,3] */
  asm volatile ("ldd [%o0+16],%i4"); /* %i4,%i5 = cs[4,5] */
  asm volatile ("ldd [%o0+24],%l0"); /* %l0,%l1 = cs[6,7] */
  asm volatile ("ldd [%o0+32],%l2"); /* %l2,%l3 = cs[8,9] */
  asm volatile ("ldd [%o0+40],%l4"); /* %l4,%l5 = cs[10,11] */
  asm volatile ("retl");
  asm volatile ("ld	[%o0+48],%l6"); /* %l6 = cs[12] */
  asm volatile ("! REAL INSN ENDS");

#elif defined(__alpha)
  asm volatile ("# REAL INSN BEGINS");
  asm volatile ("ldt $f2,0($16)"); /* $f2 = cs[0] */
  asm volatile ("ldt $f3,8($16)"); /* $f3 = cs[1] */
  asm volatile ("ldt $f4,16($16)"); /* $f4 = cs[2] */
  asm volatile ("ldt $f5,24($16)"); /* $f5 = cs[3] */
  asm volatile ("ldt $f6,32($16)"); /* $f6 = cs[4] */
  asm volatile ("ldt $f7,40($16)"); /* $f7 = cs[5] */
  asm volatile ("ldt $f8,48($16)"); /* $f8 = cs[6] */
  asm volatile ("ldt $f9,56($16)"); /* $f9 = cs[7] */
  asm volatile ("ldq $9,64($16)"); /* $9 = cs[8] */
  asm volatile ("ldq $10,72($16)"); /* $10 = cs[9] */
  asm volatile ("ldq $11,80($16)"); /* $11 = cs[10] */
  asm volatile ("ldq $12,88($16)"); /* $12 = cs[11] */
  asm volatile ("ldq $13,96($16)"); /* $13 = cs[12] */
  asm volatile ("ret $31,($26),1");
  asm volatile ("# REAL INSN ENDS");

#else

#error "unknown CPU"

#endif
}

#if HAVE_SWAP
GLOBAL long asm_swap_long(long volatile * l, long v)
{
#if defined(i386)
  asm volatile ("/ REAL INSN BEGINS");
  asm volatile ("movl 4(%esp),%ecx"); /* %ecx = l */
  asm volatile ("movl 8(%esp),%eax"); /* %eax = v */
  asm volatile ("xchg 0(%ecx),%eax"); /* swap l[0] and v */
  asm volatile ("ret");
  asm volatile ("/ REAL INSN ENDS");

#elif defined(__sparc)
  asm volatile ("! REAL INSN BEGINS");
  asm volatile ("swap [%o0],%o1");
  asm volatile ("retl");
  asm volatile ("mov %o1,%o0");
  asm volatile ("! REAL INSN ENDS");

#else

#error "unknown CPU"

#endif
  return 0;			/* to make gcc happy */
}
#endif /* HAVE_SWAP */

#if HAVE_LL_SC

/* synthesize compare and swap by LL and SC
if (*l == v) {
   swap (*l, new_v);
   return 1; 
} else return 0; 

*/
GLOBAL long asm_cmp_and_swap_long_by_ll_sc(long volatile * l, 
					   long v, long new_v)
{
#if defined(mips)
  asm volatile ("# REAL INSN BEGINS");
  asm volatile ("ll	$2,0($4)"); /* $2 = *l */
  asm volatile ("beql	$2,$5,$LCAS_TRY_SC"); /* if(*l == v) goto CAS_TRY_SC */
  asm volatile ("nop");
  asm volatile ("move	$2,$0"); /* (*l != v): return value = 0; */
  asm volatile ("j	$31");	/* return 0; */
  asm volatile ("$LCAS_TRY_SC:");
  asm volatile ("sc	$6,0($4)"); /* (delay slot): *l = new_v */
  asm volatile ("bnel	$6,$0,$LCAS_SC_OK"); /* if (succeed) goto CAS_SC_OK */
  asm volatile ("nop");
  asm volatile ("move	$2,$0"); /* (sc failed): return value = 0; */
  asm volatile ("j	$31");	/* return 0; */
  asm volatile ("$LCAS_SC_OK:");
  asm volatile ("li	$2,0x00000001"); /* return value = 1 */
  asm volatile ("j	$31");	/* return 1; */
  asm volatile ("# REAL INSN ENDS");

#elif defined(__alpha)
  asm volatile ("# REAL INSN BEGINS");
  asm volatile ("ldq_l $1,0($16)"); /* $1 = *l */
  asm volatile ("cmpeq $1,$17,$2"); /* $2 = (*l == v) */
  asm volatile ("bne $2,$CAS_TRY_SC"); /* if (*l == v) goto CAS_TRY_SC */
  asm volatile ("bis $31,$31,$0"); /* (*l != v): return value = 0; */
  asm volatile ("ret $31,($26),1");	/* return 0; */
  asm volatile ("$CAS_TRY_SC:");
  asm volatile ("stq_c $18,0($16)"); /* *l = new_v */
  asm volatile ("bne $18,$CAS_SC_OK"); /* if (succeed) goto CAS_SC_OK */
  asm volatile ("bis $31,$31,$0"); /* (sc failed): return value = 0; */
  asm volatile ("ret $31,($26),1"); /* return 0; */
  asm volatile ("$CAS_SC_OK:");
  asm volatile ("bis $31,1,$0"); /* return value = 1 */
  asm volatile ("ret $31,($26),1"); /* return 1; */
  asm volatile ("# REAL INSN ENDS");

#else

#error "unknown CPU that has ll/sc"

#endif
  return 0;			/* to make gcc happy */
}

#endif /* HAVE_LL_SC */

#if HAVE_CMP_AND_SWAP

GLOBAL long asm_cmp_and_swap_long(long volatile * l, long v, long new_v)
{
#if defined(i386)

  asm volatile ("/ REAL INSN BEGINS");
  asm volatile ("movl 4(%esp),%edx"); /* %edx = l; */
  asm volatile ("movl 8(%esp),%eax"); /* %eax = v; */
  asm volatile ("movl 12(%esp),%ecx"); /* %ecx = new_v */
  asm volatile ("lock");
  asm volatile ("cmpxchg %ecx,0(%edx)"); /* if (*l == %eax) *l = new_v; 
					    else %eax = *l; */
  asm volatile ("movl $0,%eax"); /* %eax = 0 */
  asm volatile ("sete %al");	/* if (swap_has_occurred) %eax = 1 */
  asm volatile ("ret");
  asm volatile ("/ REAL INSN ENDS");

#elif defined(__sparc)
  asm volatile ("! REAL INSN BEGINS");
  asm volatile ("cas [%o0], %o1, %o2"); /* if *l == v, swap *l and new_v */
  asm volatile ("cmp %o1,%o2");	/* new_v == v? */
  asm volatile ("beq .LCAS_OK");
  asm volatile ("mov 1,%o0");	/* if (new_v == v) return value = 1 */
  asm volatile ("mov %g0,%o0");	/* otherwise return value = 0 */
  asm volatile ("retl");
  asm volatile ("nop");
  asm volatile (".LCAS_OK:");
  asm volatile ("retl");
  asm volatile ("nop");
  asm volatile ("! REAL INSN ENDS");

#else
#error "unknown CPU that has compare-and-swap"
#endif
  return 0;			/* to make gcc happy */
}

#endif /* HAVE_CMP_AND_SWAP */

/* asm_read_and_set_lsbxx does the following;

lsb(long *l)
{
  register long v = *l;  ---(x)
  register long v1 = v | 1;
  if (v == v1) return v;
  else {
    set_v1_to_l_if_l_has_not_been_written_since_x(l, v, v1);
    if (succeed) return v;
    else return 1;
  }
}

   the lowest two bits of *l must be 
   00 : clear
   01 : available only for read
   11 : locked 

   following procedures try to perform set LSB of *l atomically.
   (i.e., *l = *l | 1).  it fails if LSB of *l is already set, 
   or read-modify-write is intervenned by other writes.
   it returns a value whose LSB is 1 when failed.
   otherwise it returns the previous value of *l, whose LSB is zero. */

#if HAVE_ASM_READ_AND_SET_LSB8
long asm_read_and_set_lsb8(void * l)
{
#if defined(i386)
#error "i386 does not have asm_read_and_set_lsb8"

#elif defined(__sparc)
  asm volatile ("! REAL INSN BEGINS");
#error "sorry not implemented yet"
  asm volatile ("! REAL INSN ENDS");

#elif defined(mips)
  asm volatile ("# REAL INSN BEGINS");
#warning "sorry not implemented yet"
  asm volatile ("# REAL INSN ENDS");

#elif defined(__alpha)
  asm volatile ("# REAL INSN BEGINS");
  asm volatile ("ldq_l $0,0($16)");	/* v = *l; */
  asm volatile ("bis $0,1,$1"); /* v1 = v | 1; */
  asm volatile ("subq $0,$1,$2"); /* tmp = v - v1; */
  asm volatile ("bne $2,$TRY_SET8"); /* if (v - v1 != 0) goto TRY_SET8; */
  asm volatile ("ret $31,($26),1");
  asm volatile ("$TRY_SET8:");
  asm volatile ("stq_c $2,0($16)"); /* *l = v1; */
  asm volatile ("bne $2,$LOCK_OK8"); /* if (succeed) goto LOCK_OK8; */
  asm volatile ("bis $31,1,$0"); /* else return 1; */
  asm volatile ("ret $31,($26),1");
  asm volatile ("$LOCK_OK8:");
  asm volatile ("ret $31,($26),1");
  asm volatile ("# REAL INSN ENDS");
  
#else
#error "wrong CPU"
#endif /* CPU */
  return 0;
}
#endif /* HAVE_ASM_READ_AND_SET_LSB8 */

#if HAVE_ASM_READ_AND_SET_LSB4
long asm_read_and_set_lsb4(void * l)
{
#if defined(i386)
  asm volatile ("/ REAL INSN BEGINS");
  asm volatile ("movl 4(%esp),%edx"); /* %edx = l */
  asm volatile ("movl 0(%edx),%eax"); /* v = *l */
  asm volatile ("movl %eax,%ecx");
  asm volatile ("orl $1,%ecx");	/* v' = v | 1 */
  asm volatile ("cmp %eax,%ecx"); /* v == v' (2 bits are set)? */
  asm volatile ("jne .LTRY_SET4"); /* if(v != v') try to write */
  asm volatile ("ret");		/* return V (LSB is 1) */
  asm volatile (".LTRY_SET4:");	/* try to write V' */
  asm volatile ("lock");
  asm volatile ("cmpxchg %ecx,0(%edx)"); /* if(*l == %eax) *l = v'; 
					    else %eax = *l; */
  asm volatile ("je .LLOCK_OK4");
  asm volatile ("movl $1,%eax"); /* say I have failed */
  asm volatile ("ret");
  asm volatile (".LLOCK_OK4:");	/* return V (LSB is 0) */
  asm volatile ("ret");
  asm volatile ("/ REAL INSN ENDS");

#elif defined(__sparc)
  asm volatile ("! REAL INSN BEGINS");
  asm volatile ("ld [%o0],%o1"); /* v = *l */
  asm volatile ("or %o1,1,%o2"); /* v1 = v || 1 */
  asm volatile ("cmp %o1,%o2");	/* v == v1? (v:1 is set?) */
  asm volatile ("bne,a .LTRY_SET4"); /* if (v:1 is zero)goto TRY_SET4; */
  asm volatile ("cas [%o0],%o1,%o2"); /* if *l == v, swap *l and new_v */
  asm volatile ("retl");		/* else return v */
  asm volatile ("mov %o1,%o0");
  asm volatile (".LTRY_SET4:");
  asm volatile ("cmp %o1,%o2");	/* v2 == v? (has swap occurred?) */
  asm volatile ("beq .LLOCK_OK4"); /* if(swap has occurred) goto .LOCKOK */
  asm volatile ("nop");		
  asm volatile ("retl");		/* else return 1; */
  asm volatile ("mov 1,%o0");
  asm volatile (".LLOCK_OK4:");
  asm volatile ("retl");		/* return v; */
  asm volatile ("mov %o1,%o0");
  asm volatile ("! REAL INSN ENDS");

#elif defined(mips)
  asm volatile ("# REAL INSN BEGINS");
  asm volatile ("ll	$5,0($4)"); /* v = *l */
  asm volatile ("ori	$3,$5,0x0001"); /* v1 = v || 1 */
  asm volatile ("bne	$5,$3,$LTRY_SET4");	/* if(v != v1 v:1 is zero?) 
					   goto $TRY_SET4 */
  asm volatile ("nop");
  asm volatile ("move	$2,$5");	/* else return v */
  asm volatile ("j	$31");	
  asm volatile ("$LTRY_SET4:");
  asm volatile ("sc	$3,0($4)");	/* *l = v1 */
  asm volatile ("bnel	$3,$0,$LLOCK_OK4"); /* if (succeed) goto $LOCK_OK4 */
  asm volatile ("nop");
  asm volatile ("li	$2,0x00000001		# 1"); /* else return 1 */
  asm volatile ("j	$31");
  asm volatile ("$LLOCK_OK4:");
  asm volatile ("move	$2,$5");	/* return v; */
  asm volatile ("j	$31");
  asm volatile ("# REAL INSN ENDS");

#elif defined(__alpha)
  asm volatile ("# REAL INSN BEGINS");
  asm volatile ("ldl_l $0,0($16)");	/* v = *l; */
  asm volatile ("bis $0,1,$1"); /* v1 = v | 1; */
  asm volatile ("subq $0,$1,$2"); /* tmp = v - v1; */
  asm volatile ("bne $2,$TRY_SET4"); /* if (v - v1 != 0) goto TRY_SET4; */
  asm volatile ("ret $31,($26),1");
  asm volatile ("$TRY_SET4:");
  asm volatile ("stl_c $2,0($16)"); /* *l = v1; */
  asm volatile ("bne $2,$LOCK_OK4"); /* if (succeed) goto LOCK_OK8; */
  asm volatile ("bis $31,1,$0"); /* else return 1; */
  asm volatile ("ret $31,($26),1");
  asm volatile ("$LOCK_OK4:");
  asm volatile ("ret $31,($26),1");
  asm volatile ("# REAL INSN ENDS");

#else
#error "wrong CPU"
#endif /* CPU */
  return 0;
}
#endif /* HAVE_ASM_READ_AND_SET_LSB4 */

#if MO_R_BYPASS_W

PUBLIC void asm_membar_read_write()
{
#if defined(__sparc)
  asm volatile ("! REAL INSN BEGINS");
  asm volatile ("retl");
  asm volatile ("membar #LoadStore");
  asm volatile ("! REAL INSN ENDS");

#elif defined(__alpha)
  asm volatile ("# REAL INSN BEGINS");
  asm volatile ("mb");
  asm volatile ("ret $31,($26),1");
  asm volatile ("# REAL INSN ENDS");

#else
#error "unknown CPU that has membar"
#endif
}

#endif /* MO_R_BYPASS_W */

#if MO_W_BYPASS_W

PUBLIC void asm_membar_write_write()
{
#if defined(__sparc)
  asm volatile ("! REAL INSN BEGINS");
  asm volatile ("retl");
  asm volatile ("membar #StoreStore");
  asm volatile ("! REAL INSN ENDS");

#elif defined(__alpha)
  asm volatile ("# REAL INSN BEGINS");
#if 1
  asm volatile ("mb");
#else
  asm volatile ("wmb");		/* error on older (OSF 3.X) systems? */
#endif
  asm volatile ("ret $31,($26),1");
  asm volatile ("# REAL INSN ENDS");

#else
#error "unknown CPU that has membar"
#endif
}

#endif /* MO_W_BYPASS_W */

#if MO_W_BYPASS_R

PUBLIC void asm_membar_write_read()
{
#if defined(__sparc)
  asm volatile ("! REAL INSN BEGINS");
  asm volatile ("retl");
  asm volatile ("membar #StoreLoad");
  asm volatile ("! REAL INSN ENDS");

#elif defined(__alpha)
  asm volatile ("# REAL INSN BEGINS");
#if 1
  asm volatile ("mb");
#else
  asm volatile ("wmb");		/* error on older (OSF 3.X) systems? */
#endif
  asm volatile ("ret $31,($26),1");
  asm volatile ("# REAL INSN ENDS");

#else
#error "unknown CPU that has membar"
#endif
}
#endif /* MO_W_BYPASS_R */

#if MO_R_BYPASS_R

PUBLIC void asm_membar_read_read()
{
#if defined(__sparc)
  asm volatile ("! REAL INSN BEGINS");
  asm volatile ("retl");
  asm volatile ("membar #LoadLoad");
  asm volatile ("! REAL INSN ENDS");

#elif defined(__alpha)
  asm volatile ("# REAL INSN BEGINS");
  asm volatile ("mb");
  asm volatile ("ret $31,($26),1");
  asm volatile ("# REAL INSN ENDS");

#else
#error "unknown CPU that has membar"
#endif
}

#endif /* MO_R_BYPASS_R */

#if defined(__sparc)

PUBLIC void asm_mflat_setjmp(void * jb)
{
#if defined(__sparc)
  asm volatile ("! REAL INSN BEGINS");
  asm volatile ("std %l0,[%sp]");
  asm volatile ("std %l2,[%sp+8]");
  asm volatile ("std %l4,[%sp+16]");
  asm volatile ("std %l6,[%sp+24]");
  asm volatile ("std %i0,[%sp+32]");
  asm volatile ("std %i2,[%sp+40]");
  asm volatile ("std %i4,[%sp+48]");
  asm volatile ("std %i6,[%sp+56]");
  asm volatile ("mov %o7,%o1");
  asm volatile ("call setjmp,0");
  asm volatile ("mov %o1,%o7");
  asm volatile ("! REAL INSN ENDS");
#else
#error "unknown CPU that has register window"
#endif
}

PUBLIC void asm_mflat_sigsetjmp(void * jb)
{
#if defined(__sparc)
  asm volatile ("! REAL INSN BEGINS");
  asm volatile ("std %l0,[%sp]");
  asm volatile ("std %l2,[%sp+8]");
  asm volatile ("std %l4,[%sp+16]");
  asm volatile ("std %l6,[%sp+24]");
  asm volatile ("std %i0,[%sp+32]");
  asm volatile ("std %i2,[%sp+40]");
  asm volatile ("std %i4,[%sp+48]");
  asm volatile ("std %i6,[%sp+56]");
  asm volatile ("mov %o7,%o1");
  asm volatile ("call sigsetjmp,0");
  asm volatile ("mov %o1,%o7");
  asm volatile ("! REAL INSN ENDS");
#else
#error "unknown CPU that has register window"
#endif
}

#if HAVE__SETJMP
PUBLIC void asm_mflat__setjmp(void * jb)
{
#if defined(__sparc)
  asm volatile ("! REAL INSN BEGINS");
  asm volatile ("std %l0,[%sp]");
  asm volatile ("std %l2,[%sp+8]");
  asm volatile ("std %l4,[%sp+16]");
  asm volatile ("std %l6,[%sp+24]");
  asm volatile ("std %i0,[%sp+32]");
  asm volatile ("std %i2,[%sp+40]");
  asm volatile ("std %i4,[%sp+48]");
  asm volatile ("std %i6,[%sp+56]");
  asm volatile ("mov %o7,%o1");
  asm volatile ("call _setjmp,0");
  asm volatile ("mov %o1,%o7");
  asm volatile ("! REAL INSN ENDS");
#else
#error "unknown CPU that has register window"
#endif
}
#endif HAVE__SETJMP

#endif
