/* 
 * st_lock.c
 */

/* 
 * Copyright (c) 1999 by Kenjiro Taura, Akinori Yonezawa. All rights reserved.
 * Copyright (c) 1999 by Yoshihiro Oyama, Toshio Endo. All rights reserved.
 * Copyright (c) 1999 by Kunio Tabata. All rights reserved.
 * Copyright (c) 1999 by Mitsubishi Research Institute.  All rights reserved.
 * Copyright (c) 1999 by Information-technology Promotion Agency.  All rights reserved.
 *
 * THIS MATERIAL IS PROVIDED AS IS, WITH ABSOLUTELY NO WARRANTY EXPRESSED
 * OR IMPLIED.  ANY USE IS AT YOUR OWN RISK.
 *
 * Permission is hereby granted to use or copy this program
 * for any purpose,  provided the above notices are retained on all copies.
 * Permission to modify the code and to distribute modified code is granted,
 * provided the above notices are retained, and a notice that the code was
 * modified is included with the above copyright notice.
 */

#include <strings.h>

#include <st.h>
#include "st_int.h"

#define SHM_N_TRIES 100
#define SHM_N_TRIES_BEFORE_SLEEP 1000

#if PTR_BYTES == 8
#define SHM_READ_AND_SET_LSB_PTR(l) asm_read_and_set_lsb8((void *)l) 
#elif PTR_BYTES == 4
#define SHM_READ_AND_SET_LSB_PTR(l) asm_read_and_set_lsb4((void *)l) 
#else
#error "wrong PTR_BYTES"
#endif

#if INT_BYTES == 4
#define SHM_READ_AND_SET_LSB_INT(l) asm_read_and_set_lsb4((void *)l) 
#else
#error "wrong INT_BYTES"
#endif

#if LONG_BYTES == 8
#define SHM_READ_AND_SET_LSB_LONG(l) asm_read_and_set_lsb8((void *)l) 
#elif LONG_BYTES == 4
#define SHM_READ_AND_SET_LSB_LONG(l) asm_read_and_set_lsb4((void *)l) 
#else
#error "wrong LONG_BYTES"
#endif

PRIVATE void spin_wait_error(void * l)
{
  fprintf(st_errout, "worker %ld : couldn't obtain lock [%p]\n",
	  tls(worker_id), l);
  st_wg_die((void *)1);
}

/* we use the lowest two bits for lock flags.
   if the bit 0 is 1, the location is available neither for read or write.
   if the bit 0 is 0 but the bit 1 is 1, the location is available only for
   read.
   the bit 0 is never set on CPUs that have cmp&swap or ll&sc.
 */

/* define a set of spin-lock primitives for int */

#define SHM_TYPE                      int
#define SHM_STRIP_TAG(x)              ST_INT_LOC_STRIP_TAG(x)
#define SHM_ATTATCH_TAG(x)            ST_INT_LOC_ATTATCH_TAG(x)

#define SHM_LOC_T                     st_int_loc_t
#define SHM_SET_LSB                   SHM_READ_AND_SET_LSB_INT
#define SHM_TRY_READ_AND_LOCK_ANY_AUX st_try_read_and_lock_any_int_aux
#define SHM_READ                      st_read_int
#define SHM_READ_AND_LOCK             st_read_and_lock_int
#define SHM_READ_AND_LOCK_ANY         st_read_and_lock_any_int
#define SHM_TRY_READ_AND_LOCK         st_try_read_and_lock_int
#define SHM_WRITE_AND_UNLOCK          st_write_and_unlock_int

#include "lock_inc.c"

#undef SHM_TYPE
#undef SHM_STRIP_TAG
#undef SHM_ATTATCH_TAG

#undef SHM_LOC_T
#undef SHM_SET_LSB
#undef SHM_TRY_READ_AND_LOCK_ANY_AUX
#undef SHM_READ
#undef SHM_READ_AND_LOCK
#undef SHM_READ_AND_LOCK_ANY
#undef SHM_TRY_READ_AND_LOCK
#undef SHM_WRITE_AND_UNLOCK


/* define a set of spin-lock primitives for long */

#define SHM_TYPE                     long
#define SHM_STRIP_TAG(x)              ST_LONG_LOC_STRIP_TAG(x)
#define SHM_ATTATCH_TAG(x)            ST_LONG_LOC_ATTATCH_TAG(x)

#define SHM_LOC_T                     st_long_loc_t
#define SHM_SET_LSB                   SHM_READ_AND_SET_LSB_LONG
#define SHM_TRY_READ_AND_LOCK_ANY_AUX st_try_read_and_lock_any_long_aux
#define SHM_READ                      st_read_long
#define SHM_READ_AND_LOCK             st_read_and_lock_long
#define SHM_READ_AND_LOCK_ANY         st_read_and_lock_any_long
#define SHM_TRY_READ_AND_LOCK         st_try_read_and_lock_long
#define SHM_WRITE_AND_UNLOCK          st_write_and_unlock_long

#include "lock_inc.c"

#undef SHM_TYPE
#undef SHM_STRIP_TAG
#undef SHM_ATTATCH_TAG

#undef SHM_LOC_T
#undef SHM_SET_LSB
#undef SHM_TRY_READ_AND_LOCK_ANY_AUX
#undef SHM_READ
#undef SHM_READ_AND_LOCK
#undef SHM_READ_AND_LOCK_ANY
#undef SHM_TRY_READ_AND_LOCK
#undef SHM_WRITE_AND_UNLOCK

/* define a set of spin-lock primitives for (aligned) pointers */

#define SHM_TYPE                      void *
#define SHM_STRIP_TAG(x)              ST_PTR_LOC_STRIP_TAG(x)
#define SHM_ATTATCH_TAG(x)            ST_PTR_LOC_ATTATCH_TAG(x)

#define SHM_LOC_T                     st_ptr_loc_t
#define SHM_SET_LSB                   SHM_READ_AND_SET_LSB_PTR
#define SHM_TRY_READ_AND_LOCK_ANY_AUX st_try_read_and_lock_any_ptr_aux
#define SHM_READ                      st_read_ptr
#define SHM_READ_AND_LOCK             st_read_and_lock_ptr
#define SHM_READ_AND_LOCK_ANY         st_read_and_lock_any_ptr
#define SHM_TRY_READ_AND_LOCK         st_try_read_and_lock_ptr
#define SHM_WRITE_AND_UNLOCK          st_write_and_unlock_ptr

#include "lock_inc.c"

#undef SHM_TYPE
#undef SHM_STRIP_TAG
#undef SHM_ATTATCH_TAG

#undef SHM_LOC_T
#undef SHM_SET_LSB
#undef SHM_TRY_READ_AND_LOCK_ANY_AUX
#undef SHM_READ
#undef SHM_READ_AND_LOCK
#undef SHM_READ_AND_LOCK_ANY
#undef SHM_TRY_READ_AND_LOCK
#undef SHM_WRITE_AND_UNLOCK

int st_fetch_and_add_int(st_int_loc_t * l, int dx)
{
  int x = st_read_and_lock_int(l);
  st_write_and_unlock_int(l, x + dx);
  return x;
}

int st_fetch_and_add_long(st_long_loc_t * l, long dx)
{
  long x = st_read_and_lock_long(l);
  st_write_and_unlock_long(l, x + dx);
  return x;
}

/* --------------------------- */

void st_read_struct(st_int_loc_t * l, void * src, void * dest, unsigned int n)
{
  while(1) {
    unsigned int v0 = st_read_int(l);
    unsigned int v1;
    bcopy(src, dest, n);
    v1 = st_read_int(l);
    if (v0 == v1) return;
  }
}

void st_read_and_lock_struct(st_int_loc_t * l, 
			     void * src, void * dest, unsigned int n)
{
  st_read_and_lock_int(l);
  bcopy(src, dest, n);
}

void st_write_and_unlock_struct(st_int_loc_t * l,
				void * src, void * dest, unsigned int n)
{
  unsigned int v = st_read_int(l);
  bcopy(src, dest, n);
  st_write_and_unlock_int(l, v + 1);
}

#if 0
/* here is an example of how to use st_xxx_struct. 
   you must associate a st_int_loc to a structure.
   it may be separate from or embedded in the structure.
   to atomically modify a structure, you first atomically copy 
   the structure, modify the copy, and atomically write the structure. 
   read-only operations can read it even if a writer is operating on
   a copy */

typedef struct point
{
  st_int_loc_t lock;
  double x, y;
} * point_t;

void move_point(point_t p, double dx, double dy)
{
  struct point c[1];
  st_read_and_lock_struct(&p->lock, p, c, sizeof(struct point));
  c->x += dx;
  c->y += dy;
  st_write_and_unlock_struct(&p->lock, c, p, sizeof(struct point));
}

void dist_point(point_t p)
{
  struct point c[1];
  st_read_struct(&p->lock, p, c, sizeof(struct point));
  return sqrt(c->x * c->x + c->y * c->y);
}

#endif
