/* 
 * st_sync.c --- synchronization primitives
 */

/* 
 * Copyright (c) 1999 by Kenjiro Taura, Akinori Yonezawa. All rights reserved.
 * Copyright (c) 1999 by Yoshihiro Oyama, Toshio Endo. All rights reserved.
 * Copyright (c) 1999 by Kunio Tabata. All rights reserved.
 * Copyright (c) 1999 by Mitsubishi Research Institute.  All rights reserved.
 * Copyright (c) 1999 by Information-technology Promotion Agency.  All rights reserved.
 *
 * THIS MATERIAL IS PROVIDED AS IS, WITH ABSOLUTELY NO WARRANTY EXPRESSED
 * OR IMPLIED.  ANY USE IS AT YOUR OWN RISK.
 *
 * Permission is hereby granted to use or copy this program
 * for any purpose,  provided the above notices are retained on all copies.
 * Permission to modify the code and to distribute modified code is granted,
 * provided the above notices are retained, and a notice that the code was
 * modified is included with the above copyright notice.
 */

#include <st.h>

#define ST_DIE_ON_ERROR 1

/* join counter */

/* initialize the counter with V (> 0) */
int st_join_counter_init(st_join_counter_t * jc, int v)
{
#if ST_DBG
  if (v < 0) {
#if ST_DIE_ON_ERROR
    fprintf(st_errout, 
	    "worker %ld : initialize counter [%p] with a negative value %d\n", 
	    tls(worker_id), jc, v);
    st_app_die(1);
#else  /* ST_DIE_ON_ERROR */
    return ST_FAULT;
#endif /* ST_DIE_ON_ERROR */
  }
#endif /* ST_DBG */

  ST_INT_LOC_INIT(&jc->count, v);
  jc->waiters = 0;
#if ST_DBG
  jc->wait_called = 0;
#endif
  MEMBAR_WRITE_WRITE();
  return 0;			/* OK */
}

/* wait for the counter to become zero */
int st_join_counter_wait(st_join_counter_t * jc)
{
  if (!ST_INT_LOC_CHECK(&jc->count, 0)) {
    int x;
    struct st_context con[1];
    struct st_context_list cell[1];
    con->valid = 0;
    cell->c = con;
    x = st_read_and_lock_int(&jc->count);
    /* here we have an exclusive access */
    if (x == 0) {
#if ST_DBG
      jc->wait_called = 1;
#endif
      st_write_and_unlock_int(&jc->count, x); /* unlock */
    } else {
#if ST_DBG
      if (x < 0) {
	st_write_and_unlock_int(&jc->count, x); /* unlock */
	fprintf(st_errout, 
		"worker %ld : read a negative value %d from counter [%p]\n", 
		tls(worker_id), x, jc);
      }
      jc->wait_called = 1;
#endif
      /* push CON on the waiters list */
      cell->next = jc->waiters;
      jc->waiters = cell;

      MEMBAR_WRITE_WRITE();
      st_write_and_unlock_int(&jc->count, x); /* unlock */
      st_suspend_thread_n(con, 1);
    }
  }
#if ST_DBG
  else {
    if (jc->wait_called == 0) jc->wait_called = 1;
  }
#endif
  return 0;
}

/* put contexts in list L in the bottom of the deque. the tailmost 
   context is put first. after all, this processor schedules
   the tailmost context first */

static void resume_contexts_list(st_context_list_t l)
{
  if (l) {
    resume_contexts_list(l->next);
    st_resume_context(l->c);
  }
}

/* decrement the counter by V */
int st_join_counter_finish_1(st_join_counter_t * jc, int v)
{
  int old_count = st_read_and_lock_int(&jc->count);
  /* we have an exclusive access to this counter */
  int new_count = old_count - v;
  if (new_count == 0) {
    st_context_list_t ws = jc->waiters;
    jc->waiters = 0;
    MEMBAR_WRITE_WRITE();
    st_write_and_unlock_int(&jc->count, 0); /* unlock */
    resume_contexts_list(ws);
    return 0;
  } else {			/* new_count is still positive */
    st_write_and_unlock_int(&jc->count, new_count); /* unlock */
#if ST_DBG
    if (new_count < 0) {
#if ST_DIE_ON_ERROR
      fprintf(st_errout, 
	      "worker %ld : counter [%p] value becomes negative (%d -> %d)\n",
	      tls(worker_id), jc, old_count, new_count);
      st_app_die(1);
      /* never reach */
#endif /* ST_DIE_ON_ERROR */
      return ST_FAULT;
    } else {
      return 0;
    }
#endif /* ST_DBG */
  }
}

/* increase the number of required finishes after it has been created.
   it is an error to add a counter after a wait has been performed */

int st_join_counter_spawn_1(st_join_counter_t * jc, int v)
{
#if ST_DBG
  if (v < 0) {
#if ST_DIE_ON_ERROR
    fprintf(st_errout, 
	    "worker %ld : negative increment %d to counter [%p]\n",
	    tls(worker_id), v, jc);
    st_app_die(1);
#else  /* ST_DIE_ON_ERROR */
    return ST_FAULT;
#endif /* ST_DIE_ON_ERROR */
  }
#endif /* ST_DBG */
  {
    int old_count = st_read_and_lock_int(&jc->count);
    /* we have an exclusive access to this counter */
    long new_count = old_count + v;
    MEMBAR_WRITE_WRITE();
    st_write_and_unlock_int(&jc->count, new_count); /* unlock */
    return 0;
  }
}

/* simply check if count is zero and no threads are waiting */
int st_join_counter_destroy(st_join_counter_t * jc)
{
#if ST_DBG
  int c = st_read_int(&jc->count);
  st_context_list_t ws = jc->waiters;
  if (c || ws) {
#if ST_DIE_ON_ERROR
    fprintf(st_errout, 
	    "worker %ld : counter == %d and waiters == %p for counter [%p]\n",
	    tls(worker_id), c, ws, jc);
    st_app_die(1);
#endif /* ST_DIE_ON_ERROR */
    return ST_FAULT;
  }
#endif /* ST_DBG */

  return 0;
}

/* semaphore */
int st_sema_init_1(st_sema_t * sem, int count, int pos_limit)
{
#if ST_DBG
  if (pos_limit < 0) {
#if ST_DIE_ON_ERROR
    fprintf(st_errout, 
	    "worker %ld : negative pos_limit %d given for semaphore [%p]\n",
	    tls(worker_id), pos_limit, sem);
    st_app_die(1);
#else  /* ST_DIE_ON_ERROR */
  return ST_FAULT;
#endif /* ST_DIE_ON_ERROR */
  }
#endif /* ST_DBG */

  ST_INT_LOC_INIT(&sem->lock, 0);
  sem->count = count;
  sem->initial_count = count;
  sem->pos_limit = pos_limit;
  sem->waiters_head = 0;
  sem->waiters_tail = 0;
  MEMBAR_WRITE_WRITE();
  return 0;
}

/* try to decrement the counter by one. if succeed, decrement the counter
   and returns a non-negative value. otherwise, DO NOT decrement the counter
   and returns a negative value */
int st_sema_trywait(st_sema_t * sem)
{
  int r;
  int x = st_try_read_and_lock_int(&sem->lock, &r);
  if (x != -1) {
    long new_count = sem->count - 1;
    if (new_count >= 0) {
      sem->count = new_count;
      st_write_and_unlock_int(&sem->lock, 0); /* unlock */
    } else {
      /* do not update counter */
      st_write_and_unlock_int(&sem->lock, 0); /* unlock */
    }
    return new_count;
  } else {
    st_write_and_unlock_int(&sem->lock, 0); /* unlock */
    /* return a negative value */
    return ST_BUSY;
  }
}

/* decrement count by one. if the count becomes negative, block on 
   the semaphore. */
int st_sema_wait(st_sema_t * sem)
{
  struct st_context c[1];
  struct st_context_list cell[1];
  int x = st_read_and_lock_int(&sem->lock);
  long new_count = sem->count - 1;
  sem->count = new_count;
  if (new_count >= 0) {
    st_write_and_unlock_int(&sem->lock, 0); /* unlock */
    return 0;
  } else {
    /* enqueue CELL into SEM */
    cell->next = 0;
    cell->c = c;
    c->valid = 0;
    if (sem->waiters_head) {
      sem->waiters_tail->next = cell;
    } else {
      sem->waiters_head = cell;
    }
    sem->waiters_tail = cell;
    st_write_and_unlock_int(&sem->lock, 0); /* unlock */
    st_suspend_thread_n(c, 1);
    return 0;
  }
}

/* increment count by one. if the count becomes zero and any 
   thread is blocking, resume one. */
int st_sema_post(st_sema_t * sem)
{
  /* lock the semaphore and increment the counter */
  int x = st_read_and_lock_int(&sem->lock);
  long new_count = sem->count + 1;
  sem->count = new_count;
  if (new_count <= 0) {
    /* if the counter is zero or negative, check if any thread is waiting */
    st_context_list_t h = sem->waiters_head;
    if (h) {
      /* a thread is waiting. resume it. */
      st_context_t c = h->c;
      sem->waiters_head = h->next;
#if ST_DBG
      /* if the queue becomes empty, we also zero tail just in case */
      if (h->next == 0) {
	sem->waiters_tail = 0;
      }
#endif
      st_write_and_unlock_int(&sem->lock, 0); /* unlock */
      st_resume_context(c);
    } else {
      st_write_and_unlock_int(&sem->lock, 0); /* unlock */
    }
  } else {
    /* new_count != 0. positive or negative */
#if ST_DBG

    /* if the value is positive, doublecheck if no threads are waiting */
    if (sem->waiters_head) {
      st_write_and_unlock_int(&sem->lock, 0); /* unlock */
#if ST_DIE_ON_ERROR
      fprintf(st_errout, 
	      "worker %ld : semaphore (or lock) [%p] seems corrupted. "
	      "count = %ld > 0, yet threads are waiting\n",
	      tls(worker_id), sem, new_count);
      st_app_die(1);
#else  /* ST_DIE_ON_ERROR */
      return ST_FAULT;
#endif /* ST_DIE_ON_ERROR */
    }

    /* check if available resources do not exceed the predetermined limit */
    if (new_count > sem->pos_limit && sem->pos_limit > 0) {
      st_write_and_unlock_int(&sem->lock, 0); /* unlock */
#if ST_DIE_ON_ERROR
      fprintf(st_errout, 
	      "worker %ld : invalid use of semaphore (or lock) [%p]. "
	      "the count becomes more than the limit (= %d)\n",
	      tls(worker_id), sem, sem->pos_limit);
      st_app_die(1);
#else  /* ST_DIE_ON_ERROR */
      return ST_FAULT;
#endif /* ST_DIE_ON_ERROR */
    }

#endif /* ST_DBG */

    st_write_and_unlock_int(&sem->lock, 0); /* unlock */
  }
  return 0;
}

/* check if all resources are returned and no threads are waiting */
int st_sema_destroy(st_sema_t * sem)
{
  if (sem->count != sem->initial_count || sem->waiters_head) {
    return ST_FAULT;
  } else {
    return 0;
  }
}

/* mutex */
int st_mutex_init(st_mutex_t * m)
{
  /* mutex is, in its simplest form, a binary semaphore that prohibits
     more than one resource */
  return st_sema_init_1(m->s, 1, 1);
}

int st_mutex_lock(st_mutex_t * m)
{
  return st_sema_wait(m->s);
}

/* return 0 on success. ST_LOCKED otherwise */
int st_mutex_trylock(st_mutex_t * m)
{
  if (st_sema_trywait(m->s) >= 0) {
    return 0;
  } else {
    return ST_BUSY;
  }
}

int st_mutex_unlock(st_mutex_t * m)
{
  return st_sema_post(m->s);
}

int st_mutex_destroy(st_mutex_t * m)
{
  return st_sema_destroy(m->s);
}

/* initialize condition variable */
int st_cond_init(st_cond_t * cv)
{
  ST_INT_LOC_INIT(&cv->lock, 0);
  cv->waiters_head = 0;
  cv->waiters_tail = 0;
  return 0;
}

/* (atomically) release M and sleep on CV */
int st_cond_wait(st_cond_t * cv, st_mutex_t * m)
{
  struct st_context c[1];
  struct st_context_list cell[1];
  int r;
  cell->next = 0;
  cell->c = c;
  c->valid = 0;
  /* enqueue */
  st_read_and_lock_int(&cv->lock);
  r = st_mutex_unlock(m);
  if (r != 0) {
    st_write_and_unlock_int(&cv->lock, 0); /* unlock */
    return r;
  } else {
    if (cv->waiters_head) {
      cv->waiters_tail->next = cell;
    } else {
      cv->waiters_head = cell;
    }
    cv->waiters_tail = cell;
    st_write_and_unlock_int(&cv->lock, 0); /* unlock */
    st_suspend_thread_n(c, 1);
    r = st_mutex_lock(m);
    if (r != 0) return r;
    else return 0;
  }
}

int st_cond_signal(st_cond_t * cv_)
{
  st_cond_t volatile * cv = (st_cond_t volatile *)cv_;

  if (cv->waiters_head) {
    st_context_list_t h;
    st_read_and_lock_int(&cv_->lock);
    h = cv->waiters_head;
    if (h) {
      cv->waiters_head = h->next;
      st_write_and_unlock_int(&cv_->lock, 0); /* unlock */
      st_resume_context(h->c);
    } else {
      st_write_and_unlock_int(&cv_->lock, 0); /* unlock */
    }
  }
  return 0;
}

int st_cond_broadcast(st_cond_t * cv_)
{
  st_cond_t volatile * cv = (st_cond_t volatile *)cv_;

  if (cv->waiters_head) {
    st_context_list_t h, p;
    st_read_and_lock_int(&cv_->lock);
    h = cv->waiters_head;
    cv->waiters_head = 0;
    st_write_and_unlock_int(&cv_->lock, 0); /* unlock */
    for (p = h; p; p = p->next) {
      st_resume_context(p->c);
    }
  }
  return 0;
}

int st_cond_destroy(st_cond_t * cv)
{
#if ST_DBG
  if (cv->waiters_head) {
#if ST_DIE_ON_ERROR
    fprintf(st_errout, 
	    "worker %ld : there are waiters when cond [%p] is destroyed\n", 
	    tls(worker_id), cv);
    st_app_die(1);
#else  /* ST_DIE_ON_ERROR */
    return ST_FAULT;
#endif /* ST_DIE_ON_ERROR */
  }
#endif /* ST_DBG */
  return 0;
}

